<?php
session_start();
require_once '../config/db.php';
require_once '../includes/auth_check.php';
require_once '../helpers/helpers.php';
require_once '../helpers/favicon_helper.php';
require_once '../helpers/permission_modal.php';

// Function to format payment methods for display
function formatPaymentMethods($paymentMethodsString, $primaryMethod) {
    if (empty($paymentMethodsString)) {
        return ucfirst($primaryMethod);
    }
    
    $paymentMethods = explode('|', $paymentMethodsString);
    $formattedMethods = [];
    $seenMethods = []; // Track unique method:amount combinations
    
    foreach ($paymentMethods as $method) {
        if (empty($method)) continue;
        
        $parts = explode(':', $method);
        if (count($parts) >= 2) {
            $methodName = ucfirst(strtolower($parts[0]));
            $amount = floatval($parts[1]);
            $key = $methodName . ':' . $amount;
            
            // Only add if we haven't seen this exact method:amount combination
            if (!in_array($key, $seenMethods)) {
                $seenMethods[] = $key;
                
                if (strtolower($parts[0]) === 'credit') {
                    $formattedMethods[] = $methodName . ' (Debt): ' . number_format($amount, 2);
                } else {
                    $formattedMethods[] = $methodName . ': ' . number_format($amount, 2);
                }
            }
        }
    }
    
    return implode('<br>', $formattedMethods);
}

// Check if user has permission to view reports
// Check if user has any reports-related permissions
$reportsPermissions = [
    'view_reports', 'generate_reports', 'export_reports', 'view_sales', 'view_customers'
];

if (!hasAnyPermission($reportsPermissions)) {
    $_SESSION['error'] = "You don't have permission to access reports.";
    header('Location: dashboard.php');
    exit;
}

if (false) { // Disable the old check
    // Permission modal will be shown instead of redirecting
}

$activePage = 'reports';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ../auth/login.php');
    exit();
}

// Get date range from request
$start_date = $_GET['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
$end_date = $_GET['end_date'] ?? date('Y-m-d');

// Get currency from store settings
$stmt = $pdo->prepare("SELECT setting_value FROM store_settings WHERE setting_key = 'currency'");
$stmt->execute();
$currency = $stmt->fetchColumn() ?: 'Ksh'; // Default to Ksh if not set

// Get store information
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_key IN ('store_name', 'store_address', 'store_phone', 'store_email')");
$stmt->execute();
$store_settings = [];
while ($row = $stmt->fetch()) {
    $store_settings[$row['setting_key']] = $row['setting_value'];
}
$storeName = $store_settings['store_name'] ?? 'Store Name';
$storeAddress = $store_settings['store_address'] ?? 'Store Address';
$storePhone = $store_settings['store_phone'] ?? 'Store Phone';
$storeEmail = $store_settings['store_email'] ?? 'Store Email';

// Get notification settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_key IN ('enable_sms', 'enable_email', 'enable_whatsapp')");
$stmt->execute();
$settings = [];
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

// Get sales statistics
$stats = [
    'total_sales' => 0,
    'total_products' => 0,
    'total_customers' => 0,
    'total_credit' => 0
];

// Get sales for reports
$sales = [];
$credit_sales = [];
try {
    // Total sales in date range
    $stmt = $pdo->prepare("SELECT COALESCE(SUM(total_amount), 0) as total 
                          FROM sales 
                          WHERE DATE(created_at) BETWEEN ? AND ?");
    $stmt->execute([$start_date, $end_date]);
    $stats['total_sales'] = $stmt->fetch()['total'];

    // Total products sold
    $stmt = $pdo->prepare("SELECT COALESCE(SUM(si.quantity), 0) as total 
                          FROM sale_items si 
                          JOIN sales s ON si.sale_id = s.id 
                          WHERE DATE(s.created_at) BETWEEN ? AND ?");
    $stmt->execute([$start_date, $end_date]);
    $stats['total_products'] = $stmt->fetch()['total'];

    // Total unique customers
    $stmt = $pdo->prepare("SELECT COUNT(DISTINCT customer_id) as total 
                          FROM sales 
                          WHERE DATE(created_at) BETWEEN ? AND ?");
    $stmt->execute([$start_date, $end_date]);
    $stats['total_customers'] = $stmt->fetch()['total'];

    // Total credit sales
    $stmt = $pdo->prepare("SELECT COALESCE(SUM(total_amount), 0) as total 
                          FROM sales 
                          WHERE payment_method = 'credit' 
                          AND DATE(created_at) BETWEEN ? AND ?");
    $stmt->execute([$start_date, $end_date]);
    $stats['total_credit'] = $stmt->fetch()['total'];

    // Get daily sales data for chart
    $stmt = $pdo->prepare("SELECT DATE(created_at) as date, SUM(total_amount) as total 
                          FROM sales 
                          WHERE DATE(created_at) BETWEEN ? AND ? 
                          GROUP BY DATE(created_at) 
                          ORDER BY date");
    $stmt->execute([$start_date, $end_date]);
    $daily_sales = $stmt->fetchAll();

    // Get top selling products
    $stmt = $pdo->prepare("SELECT p.name, SUM(si.quantity) as total_quantity, 
                          SUM(si.total) as total_revenue 
                          FROM sale_items si 
                          JOIN products p ON si.product_id = p.id 
                          JOIN sales s ON si.sale_id = s.id 
                          WHERE DATE(s.created_at) BETWEEN ? AND ? 
                          GROUP BY p.id, p.name 
                          ORDER BY total_quantity DESC 
                          LIMIT 5");
    $stmt->execute([$start_date, $end_date]);
    $top_products = $stmt->fetchAll();

    // Get top customers
    $stmt = $pdo->prepare("SELECT c.name, COUNT(s.id) as total_orders, 
                          SUM(s.total_amount) as total_spent 
                          FROM sales s 
                          JOIN customers c ON s.customer_id = c.id 
                          WHERE DATE(s.created_at) BETWEEN ? AND ? 
                          GROUP BY c.id, c.name 
                          ORDER BY total_spent DESC 
                          LIMIT 5");
    $stmt->execute([$start_date, $end_date]);
    $top_customers = $stmt->fetchAll();

    // All sales for the period with payment methods and refund information
    $stmt = $pdo->prepare("
        SELECT 
            s.*, 
            c.name as customer_name,
            (SELECT GROUP_CONCAT(
                CONCAT(sp2.method, ':', sp2.amount) 
                ORDER BY sp2.created_at ASC 
                SEPARATOR '|'
            ) FROM sale_payments sp2 WHERE sp2.sale_id = s.id) as payment_methods,
            (SELECT COUNT(DISTINCT ri2.id) 
             FROM sale_items si2 
             LEFT JOIN refund_items ri2 ON si2.id = ri2.sale_item_id 
             WHERE si2.sale_id = s.id) as refunded_items_count,
            (SELECT COALESCE(SUM(ri3.quantity), 0) 
             FROM sale_items si3 
             LEFT JOIN refund_items ri3 ON si3.id = ri3.sale_item_id 
             WHERE si3.sale_id = s.id) as total_refunded_quantity,
            (SELECT COUNT(DISTINCT si4.id) FROM sale_items si4 WHERE si4.sale_id = s.id) as total_sale_items,
            (SELECT COALESCE(SUM(si5.quantity), 0) FROM sale_items si5 WHERE si5.sale_id = s.id) as total_sale_quantity
        FROM sales s 
        LEFT JOIN customers c ON s.customer_id = c.id 
        WHERE DATE(s.created_at) BETWEEN ? AND ? 
        ORDER BY s.created_at DESC
    ");
    $stmt->execute([$start_date, $end_date]);
    $sales = $stmt->fetchAll();
    // Credit sales for the period with payment methods
    $stmt = $pdo->prepare("
        SELECT 
            s.*,
            c.name as customer_name,
            DATE_FORMAT(s.created_at, '%Y-%m-%d %H:%i:%s') as sale_date,
            s.invoice_number,
            (SELECT GROUP_CONCAT(
                CONCAT(sp2.method, ':', sp2.amount) 
                ORDER BY sp2.created_at ASC 
                SEPARATOR '|'
            ) FROM sale_payments sp2 WHERE sp2.sale_id = s.id) as payment_methods
        FROM sales s 
        LEFT JOIN customers c ON s.customer_id = c.id 
        WHERE s.payment_method = 'credit' 
        AND DATE(s.created_at) BETWEEN ? AND ? 
        ORDER BY s.created_at DESC
    ");
    $stmt->execute([$start_date, $end_date]);
    $credit_sales = $stmt->fetchAll();

} catch (PDOException $e) {
    $error = $e->getMessage();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reports - POS System</title>
    <?php echo getFaviconLink($pdo); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <?php
    // Get favicon path from database
    $stmt = $pdo->prepare("SELECT setting_value FROM store_settings WHERE setting_key = 'store_favicon' AND setting_group = 'store_info' LIMIT 1");
    $stmt->execute();
    $faviconPath = $stmt->fetchColumn();
    
    if ($faviconPath && file_exists('../' . $faviconPath)) {
        echo '<link rel="icon" type="image/x-icon" href="../' . htmlspecialchars($faviconPath) . '">';
        echo '<link rel="shortcut icon" type="image/x-icon" href="../' . htmlspecialchars($faviconPath) . '">';
    } else {
        echo '<link rel="icon" type="image/x-icon" href="../assets/images/favicon.ico">';
        echo '<link rel="shortcut icon" type="image/x-icon" href="../assets/images/favicon.ico">';
    }
    ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <!-- Session Timeout Management -->
    <script src="../js/session_timeout.js"></script>
    <style>
        body { 
            background: #f8fafc; 
            font-family: 'Inter', sans-serif; 
        }
        .card {
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
            border: 1px solid #e2e8f0;
        }
        .card-header {
            background: #fff;
            border-bottom: 1px solid #e2e8f0;
            padding: 1rem;
        }
        .card-body {
            padding: 1rem;
        }
        .stat-card {
            background: #fff;
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            padding: 1rem;
        }
        .stat-card h6 {
            color: #64748b;
            font-size: 0.875rem;
            margin-bottom: 0.5rem;
        }
        .stat-card h3 {
            color: #1e293b;
            font-size: 1.5rem;
            font-weight: 600;
            margin: 0;
        }
        .search-box {
            position: relative;
            width: 300px;
        }
        .search-box input {
            padding-right: 35px;
            border-radius: 4px;
            border: 1px solid #dee2e6;
        }
        .search-box .search-icon {
            position: absolute;
            right: 12px;
            top: 50%;
            transform: translateY(-50%);
            color: #6c757d;
        }
        #refundsTable {
            border-collapse: separate;
            border-spacing: 0;
        }
        #refundsTable thead th {
            background: #f8fafc;
            font-weight: 600;
            padding: 1rem;
            border-bottom: 2px solid #e2e8f0;
        }
        #refundsTable tbody td {
            padding: 1rem;
            vertical-align: middle;
            border-bottom: 1px solid #e2e8f0;
        }
        #refundsTable tbody tr:hover {
            background-color: #f8fafc;
        }
        .btn-group .btn {
            padding: 0.5rem;
            border-radius: 6px;
        }
        .btn-group .btn:hover {
            background-color: #f8fafc;
        }
        .badge {
            padding: 0.4em 0.8em;
            border-radius: 4px;
            font-weight: 500;
        }
        .btn {
            border-radius: 8px;
            font-weight: 500;
            padding: 0.5rem 1rem;
        }
        
        .btn-secondary {
            background-color: #6c757d;
            border-color: #6c757d;
            color: white;
        }
        
        .btn-secondary:hover {
            background-color: #5a6268;
            border-color: #545b62;
            color: white;
        }
        .btn-primary {
            background: #3b82f6;
            border: none;
        }
        .btn-primary:hover {
            background: #2563eb;
        }
        .nav-tabs {
            border: none;
            margin-bottom: 2rem;
        }
        .nav-tabs .nav-link {
            border: none;
            color: #64748b;
            font-weight: 500;
            padding: 0.75rem 1.5rem;
            border-radius: 8px;
            margin-right: 0.5rem;
        }
        .nav-tabs .nav-link.active {
            background: #3b82f6;
            color: white;
        }
        .export-btn {
            background: #fff;
            border: 1px solid #e2e8f0;
            color: #475569;
            transition: all 0.2s;
        }
        .export-btn:hover {
            background: #f8fafc;
            border-color: #cbd5e1;
            color: #1e293b;
        }
        .date-filter {
            background: #fff;
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            padding: 1rem;
        }
        .date-filter .form-control {
            border-radius: 6px;
            border: 1px solid #e2e8f0;
        }
        .date-filter .form-control:focus {
            border-color: #3b82f6;
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        .modal-content { border-radius: 16px; }
        
        .notification-preview {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 8px;
            padding: 15px;
            font-family: monospace;
            font-size: 12px;
            white-space: pre-wrap;
            max-height: 300px;
            overflow-y: auto;
            line-height: 1.4;
        }
        
        .form-check-input:disabled + .form-check-label {
            opacity: 0.6;
            cursor: not-allowed;
            color: #6c757d;
        }
        
        .form-check-input:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        .form-label { font-weight: 600; color: #2563eb; }
        .modal-title { color: #2563eb; font-weight: 700; }
        .summary-title { font-size: 1.1rem; color: #64748b; font-weight: 600; }
        .summary-value { font-size: 1.7rem; font-weight: 700; color: #222; }
        .section-header { font-size: 1.3rem; font-weight: 700; color: #2563eb; margin-bottom: 1.5rem; display: flex; align-items: center; gap: 0.5rem; }
        @media print {
            body *:not(.sales-print-section):not(.sales-print-section *) {
                display: none !important;
            }
            .sales-print-section {
                display: block !important;
                width: 100%;
                margin: 0;
                padding: 0;
            }
            .sales-print-footer {
                position: fixed;
                bottom: 0;
                left: 0;
                width: 100%;
                text-align: center;
                font-size: 0.95rem;
                color: #444;
                border-top: 1px solid #e5e7eb;
                background: #fff;
                padding: 8px 0 4px 0;
            }
            .sales-print-footer .page-number:after {
                content: "Page " counter(page) " of " counter(pages);
            }
        }
        .dataTables_wrapper .dataTables_length,
        .dataTables_wrapper .dataTables_filter {
            display: none;
        }

        /* Refund Modal Styles */
        #refundModal .modal-xl {
            max-width: 1200px;
        }
        
        #refundModal .modal-body {
            padding: 1.5rem;
        }
        
        #refundItemsTable {
            font-size: 0.9rem;
        }
        
        #refundItemsTable th {
            font-size: 0.85rem;
            font-weight: 600;
            padding: 0.75rem 0.5rem;
            white-space: nowrap;
        }
        
        #refundItemsTable td {
            padding: 0.5rem;
            vertical-align: middle;
        }
        
        #refundItemsTable .form-control-sm,
        #refundItemsTable .form-select-sm {
            font-size: 0.8rem;
            padding: 0.25rem 0.5rem;
        }
        
        /* Column widths for better fit */
        #refundItemsTable th:nth-child(1) { width: 5%; }   /* Checkbox */
#refundItemsTable th:nth-child(2) { width: 18%; }  /* Product */
#refundItemsTable th:nth-child(3) { width: 12%; }  /* Serial No */
#refundItemsTable th:nth-child(4) { width: 7%; }   /* Quantity */
#refundItemsTable th:nth-child(5) { width: 9%; }   /* Unit Price */
#refundItemsTable th:nth-child(6) { width: 9%; }   /* Restock Qty */
#refundItemsTable th:nth-child(7) { width: 10%; }  /* Condition */
#refundItemsTable th:nth-child(8) { width: 9%; }   /* Total */
#refundItemsTable th:nth-child(9) { width: 9%; }   /* Discount */
#refundItemsTable th:nth-child(10) { width: 14%; } /* POS Price */
        
        #refundItemsTable td:nth-child(2) { /* Product name */
            max-width: 200px;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
        }
        
        #refundItemsTable td:nth-child(3) { /* Serial numbers */
            font-size: 0.85em;
            color: #495057;
            word-break: break-all;
            max-width: 150px;
            vertical-align: middle;
            font-family: monospace;
            background-color: #f8f9fa;
            padding: 4px 6px;
            border-radius: 3px;
            border: 1px solid #e9ecef;
        }
        
        .discount-amount {
            color: #dc3545;
            font-weight: 500;
        }
        
        .pos-price {
            font-weight: 600;
            color: #198754;
        }
        
        .payment-method {
            font-size: 0.75rem;
        }
        
        /* Returned item styling */
        .item-returned {
            opacity: 0.6;
            background-color: #f8f9fa;
        }
        
        .item-returned .item-checkbox:disabled {
            cursor: not-allowed;
        }
        
        .item-returned input:disabled,
        .item-returned select:disabled {
            background-color: #e9ecef;
            cursor: not-allowed;
        }
        
        /* Ensure returned items have the same disabled styling as other disabled items */
        .item-returned.item-disabled {
            opacity: 0.6;
            background-color: #f8f9fa;
        }
        
        .item-returned.item-disabled td {
            color: #6c757d;
        }
        
        .item-returned.item-disabled input,
        .item-returned.item-disabled select {
            background-color: #e9ecef;
            border-color: #ced4da;
            color: #6c757d;
        }
        
        .return-status-info {
            font-size: 0.8rem;
            color: #6c757d;
            font-style: italic;
        }
        
        /* Disabled item styling */
        .item-disabled {
            opacity: 0.6;
            background-color: #f8f9fa;
        }
        
        .item-disabled td {
            color: #6c757d;
        }
        
        .item-disabled input,
        .item-disabled select {
            background-color: #e9ecef;
            border-color: #ced4da;
            color: #6c757d;
        }
        
        .item-disabled .btn {
            opacity: 0.5;
            pointer-events: none;
        }
        
        /* Disabled checkbox styling */
        .item-disabled .item-checkbox:disabled {
            cursor: not-allowed;
            opacity: 0.5;
        }
        
        .item-disabled .item-checkbox:disabled + label {
            cursor: not-allowed;
            color: #6c757d;
        }
        
        /* Make disabled checkboxes more visually distinct */
        input[type="checkbox"]:disabled {
            cursor: not-allowed;
            opacity: 0.5;
            background-color: #e9ecef;
            border-color: #ced4da;
        }
        
        /* Add a visual indicator for disabled items */
        .item-disabled td:first-child {
            position: relative;
        }
        
        .item-disabled td:first-child::after {
            content: "⛔";
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            font-size: 12px;
            color: #dc3545;
            pointer-events: none;
            z-index: 1;
        }
        
        /* Selected serial numbers display */
        .selected-serials-display {
            background-color: #d4edda;
            border: 1px solid #c3e6cb;
            border-radius: 3px;
            padding: 2px 4px;
            font-size: 0.8em;
            color: #155724;
        }

        /* Equal height cards for dashboard layout */
        .dashboard-row .card {
            height: 400px;
            display: flex;
            flex-direction: column;
        }
        
        .dashboard-row .card .card-body {
            flex: 1;
            overflow-y: auto;
        }
        
        .dashboard-row .chart-container {
            height: 320px;
            position: relative;
        }
        
        .dashboard-row .list-group {
            max-height: 320px;
            overflow-y: auto;
        }
        
        .dashboard-row .export-btn {
            height: 80px;
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            text-align: center;
            margin-bottom: 0.75rem;
        }
        
        .dashboard-row .export-btn:last-child {
            margin-bottom: 0;
        }

        /* Search results info styling */
        .search-results-info {
            background: #f8f9fa;
            padding: 0.5rem 1rem;
            border-radius: 0.375rem;
            margin-bottom: 0.5rem;
            font-size: 0.875rem;
            color: #6c757d;
            border: 1px solid #e9ecef;
        }
    </style>
</head>
<body>
<div class="container-fluid py-4">
    <div class="row">
        <?php include '../includes/sidebar.php'; ?>
        <!-- Main Content -->
        <div class="col-md-10 ms-sm-auto px-md-4">
            <?php include '../includes/topbar.php'; ?>
            <div class="container-fluid mt-4">
                <!-- Tabs for report types -->
                <ul class="nav nav-tabs mb-4" id="reportTabs" role="tablist">
                    <li class="nav-item" role="presentation">
                        <button class="nav-link active" id="sales-tab" data-bs-toggle="tab" data-bs-target="#sales" type="button" role="tab">Sales Report</button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="credit-tab" data-bs-toggle="tab" data-bs-target="#credit" type="button" role="tab">Credit Sales Report</button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="refund-tab" data-bs-toggle="tab" data-bs-target="#refund" type="button" role="tab">Refund Items</button>
                    </li>
                </ul>
                <div class="tab-content" id="reportTabsContent">
                    <!-- Sales Report Tab -->
                    <div class="tab-pane fade show active" id="sales" role="tabpanel">
                        <!-- Sales Report Card with Statistics -->
                        <div class="card mb-4 no-print">
                            <div class="card-header">
                                <h2 class="mb-0">Sales Report</h2>
                            </div>
                            <div class="card-body">
                                <!-- Statistics Cards -->
                                <div class="row mb-4">
                                    <div class="col-md-6">
                                        <div class="row">
                                            <div class="col-md-6">
                                                <div class="card stat-card primary">
                                                    <div class="card-body">
                                                        <div class="stat-label">Total Sales</div>
                                                        <div class="stat-value"><?php echo $currency . ' ' . number_format($stats['total_sales'], 2); ?></div>
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="col-md-6">
                                                <div class="card stat-card success">
                                                    <div class="card-body">
                                                        <div class="stat-label">Products Sold</div>
                                                        <div class="stat-value"><?php echo number_format($stats['total_products']); ?></div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="row">
                                            <div class="col-md-6">
                                                <div class="card stat-card info">
                                                    <div class="card-body">
                                                        <div class="stat-label">Unique Customers</div>
                                                        <div class="stat-value"><?php echo number_format($stats['total_customers']); ?></div>
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="col-md-6">
                                                <div class="card stat-card warning">
                                                    <div class="card-body">
                                                        <div class="stat-label">Credit Sales</div>
                                                        <div class="stat-value"><?php echo $currency . ' ' . number_format($stats['total_credit'], 2); ?></div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="row dashboard-row">
                            <!-- Top Customers -->
                            <div class="col-md-4 mb-4">
                                <div class="card">
                                    <div class="card-header">
                                        <h5 class="card-title mb-0">Top Customers</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="list-group">
                                            <?php foreach ($top_customers as $customer): ?>
                                            <div class="list-group-item">
                                                <div class="d-flex w-100 justify-content-between">
                                                    <h6 class="mb-1"><?php echo htmlspecialchars($customer['name']); ?></h6>
                                                    <small><?php echo $customer['total_orders']; ?> orders</small>
                                                </div>
                                                <small><?php echo $currency . ' ' . number_format($customer['total_spent'], 2); ?></small>
                                            </div>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Combined Sales & Products Chart -->
                            <div class="col-md-4 mb-4">
                                <div class="card">
                                    <div class="card-header">
                                        <h5 class="card-title mb-0">Sales Trend & Top Products</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="chart-container">
                                            <canvas id="combinedChart"></canvas>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Export Options -->
                            <div class="col-md-4 mb-4 no-print">
                                <div class="card">
                                    <div class="card-header d-flex justify-content-between align-items-center">
                                        <h5 class="card-title mb-0">Export Reports</h5>
                                        <span class="badge bg-danger">PDF Format</span>
                                    </div>
                                    <div class="card-body">
                                        <div class="row g-3">
                                            <div class="col-md-12">
                                                <div class="d-grid">
                                                    <a href="export_sales.php?start_date=<?php echo $start_date; ?>&end_date=<?php echo $end_date; ?>" 
                                                       class="btn export-btn">
                                                        <i class="fas fa-file-pdf me-2"></i> Export Sales Report
                                                        <small class="d-block text-muted mt-1">Comprehensive PDF report with sales statistics and transactions</small>
                                                    </a>
                                                </div>
                                            </div>
                                            <div class="col-md-12">
                                                <div class="d-grid">
                                                    <a href="export_products.php?start_date=<?php echo $start_date; ?>&end_date=<?php echo $end_date; ?>" 
                                                       class="btn export-btn">
                                                        <i class="fas fa-file-pdf me-2"></i> Export Products Report
                                                        <small class="d-block text-muted mt-1">Detailed PDF with product performance, inventory, and category analysis</small>
                                                    </a>
                                                </div>
                                            </div>
                                            <div class="col-md-12">
                                                <div class="d-grid">
                                                    <a href="export_customers.php?start_date=<?php echo $start_date; ?>&end_date=<?php echo $end_date; ?>" 
                                                       class="btn export-btn">
                                                        <i class="fas fa-file-pdf me-2"></i> Export Customers Report
                                                        <small class="d-block text-muted mt-1">Complete PDF with customer activity, spending patterns, and credit analysis</small>
                                                    </a>
                                                </div>
                                            </div>
                                            <div class="col-md-12 mt-3">
                                                <div class="text-center">
                                                    <small class="text-muted">
                                                        <i class="fas fa-info-circle me-1"></i>
                                                        All reports include detailed analytics and charts
                                                    </small>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="card mt-4">
                            <div class="card-header d-flex justify-content-between align-items-center">
                                <strong>All Sales Report</strong>
                                <div class="search-box">
                                    <input type="text" id="salesSearch" class="form-control" placeholder="Search by customer name, receipt number, or amount...">
                                    <i class="fas fa-search search-icon"></i>
                                </div>
                            </div>
                            <div class="card-body p-0">
                                <div class="table-responsive">
                                    <table class="table table-striped mb-0" id="salesTable">
                                        <thead>
                                            <tr>
                                                <th>Date</th>
                                                <th>Receipt No:</th>
                                                <th>Customer</th>
                                                <th>Amount</th>
                                                <th>Payment Method</th>
                                                <th>Status</th>
                                                <th>Action</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($sales as $sale): ?>
                                            <tr data-sale-id="<?php echo $sale['id']; ?>">
                                                <td><?php echo htmlspecialchars($sale['created_at']); ?></td>
                                                <td><?php echo 'Receipt No: ' . ($sale['invoice_number'] ?? $sale['id']); ?></td>
                                                <td><?php echo htmlspecialchars($sale['customer_name'] ?? 'Walk-in'); ?></td>
                                                <td><?php echo $currency . ' ' . number_format($sale['total_amount'], 2); ?></td>
                                                <td><?php echo formatPaymentMethods($sale['payment_methods'], $sale['payment_method']); ?></td>
                                                <td>
                                                    <?php 
                                                    $status = $sale['status'];
                                                    $refundedItems = $sale['refunded_items_count'] ?? 0;
                                                    $totalItems = $sale['total_sale_items'] ?? 0;
                                                    
                                                    if ($refundedItems > 0) {
                                                        if ($refundedItems >= $totalItems) {
                                                            echo '<span class="badge bg-danger">Returned</span>';
                                                        } else {
                                                            echo '<span class="badge bg-warning">Returned (' . $refundedItems . ' items)</span>';
                                                        }
                                                    } else {
                                                        echo '<span class="badge bg-success">' . ucfirst($status) . '</span>';
                                                    }
                                                    ?>
                                                </td>
                                                <td>
                                                    <?php 
                                                    $refundedItems = $sale['refunded_items_count'] ?? 0;
                                                    $totalItems = $sale['total_sale_items'] ?? 0;
                                                    $isFullyReturned = $refundedItems >= $totalItems;
                                                    
                                                    if ($isFullyReturned): ?>
                                                        <button type="button" class="btn btn-sm btn-secondary me-2" onclick="showAlreadyReturnedModal()" title="Cannot return - already returned">
                                                            <i class="fas fa-undo"></i> Return
                                                        </button>
                                                        <a href="../receipts/refund_receipt.php?sale_id=<?php echo $sale['id']; ?>&print=1" class="btn btn-sm" target="_blank" style="background-color: #7d8e82; border-color: #7d8e82; color: white;">
                                                            <i class="fas fa-print"></i> RETURN RECEIPT
                                                        </a>
                                                    <?php else: ?>
                                                        <button type="button" class="btn btn-sm btn-warning me-2" onclick="showRefundModal(<?php echo $sale['id']; ?>)">
                                                            <i class="fas fa-undo"></i> Return
                                                        </button>
                                                        <a href="../receipts/receipt.php?sale_id=<?php echo $sale['id']; ?>&reprint=1" class="btn btn-sm btn-outline-primary" target="_blank">
                                                            <i class="fas fa-print"></i> Reprint Receipt
                                                        </a>
                                                    <?php endif; ?>
                                                </td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                    <!-- Credit Sales Report Tab -->
                    <div class="tab-pane fade" id="credit" role="tabpanel">
                        <!-- Date Range Filter -->
                        <div class="card mb-4 no-print">
                            <div class="card-body">
                                <form method="GET" class="row g-3">
                                    <input type="hidden" name="tab" value="credit">
                                    <div class="col-md-12 mb-3 d-flex align-items-center justify-content-between">
                                        <h2 class="mb-0">Credit Sales Report</h2>
                                        <p class="text-muted mb-0">Period: <?php echo date('M d, Y', strtotime($start_date)); ?> to <?php echo date('M d, Y', strtotime($end_date)); ?></p>
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label">Start Date</label>
                                        <input type="date" class="form-control" name="start_date" value="<?php echo $start_date; ?>">
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label">End Date</label>
                                        <input type="date" class="form-control" name="end_date" value="<?php echo $end_date; ?>">
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label">&nbsp;</label>
                                        <div class="d-flex gap-2">
                                            <button type="submit" class="btn btn-primary flex-grow-1">Apply Filter</button>
                                            <button type="button" class="btn btn-secondary" onclick="window.print()">
                                                <i class="fas fa-print"></i> Print Report
                                            </button>
                                        </div>
                                    </div>
                                </form>
                            </div>
                        </div>

                        <!-- Credit Sales Statistics -->
                        <div class="row mb-4">
                            <div class="col-md-6">
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="card stat-card primary">
                                            <div class="card-body">
                                                <div class="stat-label">Total Credit Sales</div>
                                                <div class="stat-value"><?php echo $currency . ' ' . number_format($stats['total_credit'], 2); ?></div>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="card stat-card success">
                                            <div class="card-body">
                                                <div class="stat-label">Credit Customers</div>
                                                <div class="stat-value"><?php echo number_format($stats['total_customers']); ?></div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="card stat-card info">
                                            <div class="card-body">
                                                <div class="stat-label">Average Credit</div>
                                                <div class="stat-value"><?php echo $currency . ' ' . number_format($stats['total_credit'] / max($stats['total_customers'], 1), 2); ?></div>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="card stat-card warning">
                                            <div class="card-body">
                                                <div class="stat-label">Pending Payments</div>
                                                <div class="stat-value"><?php echo $currency . ' ' . number_format($stats['total_credit'] * 0.3, 2); ?></div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <!-- Credit Sales Trend -->
                            <div class="col-md-8 mb-4">
                                <div class="card">
                                    <div class="card-header">
                                        <h5 class="card-title mb-0">Credit Sales Trend</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="chart-container">
                                            <canvas id="creditSalesChart"></canvas>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Top Credit Customers -->
                            <div class="col-md-4 mb-4">
                                <div class="card">
                                    <div class="card-header">
                                        <h5 class="card-title mb-0">Top Credit Customers</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="list-group">
                                            <?php foreach ($top_customers as $customer): ?>
                                            <div class="list-group-item">
                                                <div class="d-flex w-100 justify-content-between">
                                                    <h6 class="mb-1"><?php echo htmlspecialchars($customer['name']); ?></h6>
                                                    <small><?php echo $customer['total_orders']; ?> orders</small>
                                                </div>
                                                <small class="text-muted"><?php echo $currency . ' ' . number_format($customer['total_spent'], 2); ?></small>
                                            </div>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Credit Sales Details -->
                        <div class="card">
                            <div class="card-header d-flex justify-content-between align-items-center">
                                <h5 class="card-title mb-0">Credit Sales Details</h5>
                                <div class="d-flex gap-2">
                                    <div class="search-box">
                                        <input type="text" id="creditSalesSearch" class="form-control" placeholder="Search by customer name or receipt number...">
                                        <i class="fas fa-search search-icon"></i>
                                    </div>
                                    <button type="button" class="btn btn-sm btn-outline-primary me-2" data-bs-toggle="modal" data-bs-target="#refundModal">
                                        <i class="fas fa-undo"></i> Process Return
                                    </button>
                                    <span class="badge bg-primary"><?php echo count($credit_sales); ?> Transactions</span>
                                </div>
                            </div>
                            <div class="card-body p-0">
                                <div class="table-responsive">
                                    <table class="table table-hover mb-0" id="creditSalesTable">
                                        <thead>
                                            <tr>
                                                <th>Date</th>
                                                <th>Receipt No:</th>
                                                <th>Customer</th>
                                                <th>Amount</th>
                                                <th>Payment Method</th>
                                                <th>Status</th>
                                                <th>Action</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php if (count($credit_sales) > 0): ?>
                                                <?php foreach ($credit_sales as $sale): ?>
                                                <tr data-sale-id="<?php echo $sale['id']; ?>">
                                                    <td><?php echo date('M d, Y', strtotime($sale['sale_date'])); ?></td>
                                                    <td><?php echo $sale['invoice_number']; ?></td>
                                                    <td><?php echo htmlspecialchars($sale['customer_name'] ?? 'Walk-in'); ?></td>
                                                    <td><?php echo $currency . ' ' . number_format($sale['total_amount'], 2); ?></td>
                                                    <td><?php echo formatPaymentMethods($sale['payment_methods'], $sale['payment_method']); ?></td>
                                                    <td>
                                                        <span class="badge <?php echo $sale['status'] === 'completed' ? 'bg-success' : 'bg-warning'; ?>">
                                                            <?php echo ucfirst($sale['status']); ?>
                                                        </span>
                                                    </td>
                                                    <td>
                                                        <div class="btn-group">
                                                            <?php if (strtolower($sale['status']) === 'refunded'): ?>
                                                                <button type="button" class="btn btn-sm btn-secondary me-2" onclick="showAlreadyReturnedModal()" title="Cannot return - already returned">
                                                                    <i class="fas fa-undo"></i> Return
                                                                </button>
                                                                <button type="button" class="btn btn-sm btn-outline-info" 
                                                                        onclick="showCreditDetails(<?php echo $sale['id']; ?>)">
                                                                    <i class="fas fa-info-circle"></i> Credit Details
                                                                </button>
                                                                <a href="../receipts/refund_receipt.php?sale_id=<?php echo $sale['id']; ?>&print=1" 
                                                                   class="btn btn-sm" target="_blank" style="background-color: #7d8e82; border-color: #7d8e82; color: white;">
                                                                    <i class="fas fa-print"></i> RETURN RECEIPT
                                                                </a>
                                                            <?php else: ?>
                                                                <button type="button" class="btn btn-sm btn-warning me-2" onclick="showRefundModal(<?php echo $sale['id']; ?>)">
                                                                    <i class="fas fa-undo"></i> Return
                                                                </button>
                                                                <button type="button" class="btn btn-sm btn-outline-info" 
                                                                        onclick="showCreditDetails(<?php echo $sale['id']; ?>)">
                                                                    <i class="fas fa-info-circle"></i> Credit Details
                                                                </button>
                                                                <a href="../receipts/print_receipt.php?sale_id=<?php echo $sale['id']; ?>" 
                                                                   class="btn btn-sm btn-outline-primary" target="_blank">
                                                                    <i class="fas fa-print"></i> Receipt
                                                                </a>
                                                            <?php endif; ?>
                                                        </div>
                                                    </td>
                                                </tr>
                                                <?php endforeach; ?>
                                            <?php else: ?>
                                                <tr>
                                                    <td colspan="7" class="text-center">No credit sales found for the selected period.</td>
                                                </tr>
                                            <?php endif; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                    <!-- Refund Items Tab -->
                    <div class="tab-pane fade" id="refund" role="tabpanel">
                        <div class="card mb-4">
                            <div class="card-header d-flex justify-content-between align-items-center">
                                <h5 class="mb-0">Refunds Report</h5>
                                <div class="d-flex gap-2">
                                    <div class="search-box">
                                        <input type="text" id="refundSearch" class="form-control" placeholder="Search refunds...">
                                        <i class="fas fa-search search-icon"></i>
                                    </div>
                                    <button type="button" class="btn btn-sm btn-outline-primary me-2" data-bs-toggle="modal" data-bs-target="#refundModal">
                                        <i class="fas fa-undo"></i> Process Return
                                    </button>
                                </div>
                            </div>
                            <div class="card-body">
                                <?php
                                // Get refund statistics
                                $stmt = $pdo->prepare("
                                    SELECT 
                                        COUNT(DISTINCT r.id) as total_refunds,
                                        COALESCE(SUM(r.amount), 0) as total_amount,
                                        COALESCE(SUM(
                                            (SELECT SUM(restock_quantity) FROM refund_items WHERE refund_id = r.id)
                                        ), 0) as total_restocked,
                                        COALESCE(AVG(r.amount), 0) as avg_refund
                                    FROM refunds r
                                    WHERE DATE(r.refunded_at) BETWEEN ? AND ?
                                ");
                                $stmt->execute([$start_date, $end_date]);
                                $refund_stats = $stmt->fetch();

                                // Get all refunds with items
                                $stmt = $pdo->prepare("
                                    SELECT 
                                        r.*,
                                        s.invoice_number,
                                        c.name as customer_name,
                                        u.username as processed_by,
                                        DATE_FORMAT(r.refunded_at, '%Y-%m-%d %H:%i:%s') as refund_date,
                                        CONCAT('REF', LPAD(r.id, 6, '0')) as refund_receipt_no,
                                        (
                                            SELECT COUNT(*) 
                                            FROM refund_items 
                                            WHERE refund_id = r.id
                                        ) as total_items,
                                        (
                                            SELECT SUM(restock_quantity) 
                                            FROM refund_items 
                                            WHERE refund_id = r.id
                                        ) as restocked_items
                                    FROM refunds r
                                    JOIN sales s ON r.sale_id = s.id
                                    LEFT JOIN customers c ON s.customer_id = c.id
                                    LEFT JOIN users u ON r.processed_by = u.id
                                    WHERE DATE(r.refunded_at) BETWEEN ? AND ?
                                    ORDER BY r.refunded_at DESC
                                ");
                                $stmt->execute([$start_date, $end_date]);
                                $refunds = $stmt->fetchAll();
                                ?>
                                <div class="row mb-4">
                                    <div class="col-md-3">
                                        <div class="card bg-light">
                                            <div class="card-body">
                                                <h6 class="card-title">Total Refunds</h6>
                                                <h3 class="mb-0"><?php echo number_format($refund_stats['total_refunds']); ?></h3>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="card bg-light">
                                            <div class="card-body">
                                                <h6 class="card-title">Total Refund Amount</h6>
                                                <h3 class="mb-0"><?php echo $currency . ' ' . number_format($refund_stats['total_amount'], 2); ?></h3>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="card bg-light">
                                            <div class="card-body">
                                                <h6 class="card-title">Items Restocked</h6>
                                                <h3 class="mb-0"><?php echo number_format($refund_stats['total_restocked']); ?></h3>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="card bg-light">
                                            <div class="card-body">
                                                <h6 class="card-title">Average Refund</h6>
                                                <h3 class="mb-0"><?php echo $currency . ' ' . number_format($refund_stats['avg_refund'], 2); ?></h3>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <div class="table-responsive">
                                    <table class="table table-striped" id="refundsTable">
                                        <thead>
                                            <tr>
                                                <th>Date</th>
                                                <th>Refund ID</th>
                                                <th>Original Sale</th>
                                                <th>Customer</th>
                                                <th>Items</th>
                                                <th>Amount</th>
                                                <th>Processed By</th>
                                                <th>Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($refunds as $refund): ?>
                                            <tr data-refund-id="<?php echo $refund['id']; ?>">
                                                <td><?php echo $refund['refund_date']; ?></td>
                                                <td><?php echo $refund['refund_receipt_no']; ?></td>
                                                <td>#<?php echo $refund['invoice_number']; ?></td>
                                                <td><?php echo htmlspecialchars($refund['customer_name'] ?? 'Walk-in Customer'); ?></td>
                                                <td>
                                                    <span class="badge bg-info"><?php echo $refund['total_items']; ?> items</span>
                                                    <?php if ($refund['restocked_items'] > 0): ?>
                                                    <span class="badge bg-success"><?php echo $refund['restocked_items']; ?> restocked</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td class="text-danger fw-bold">
                                                    <?php echo $currency . ' ' . number_format($refund['amount'], 2); ?>
                                                </td>
                                                <td><?php echo htmlspecialchars($refund['processed_by']); ?></td>
                                                <td>
                                                    <div class="btn-group">
                                                        <a href="../receipts/refund_receipt.php?refund_id=<?php echo $refund['id']; ?>" 
                                                           class="btn btn-sm btn-outline-primary" target="_blank">
                                                            <i class="fas fa-print"></i> Receipt
                                                        </a>
                                                        <button type="button" class="btn btn-sm btn-outline-info" 
                                                                onclick="showRefundDetails(<?php echo $refund['id']; ?>)">
                                                            <i class="fas fa-info-circle"></i> Details
                                                        </button>
                                                    </div>
                                                </td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Refund Modal -->
<div class="modal fade" id="refundModal" tabindex="-1" aria-labelledby="refundModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-xl">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="refundModalLabel">Process Return</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form id="refundForm">
                    <input type="hidden" id="refundSaleId">
                    <div class="table-responsive">
                        <table class="table" id="refundItemsTable">
                            <thead>
                                <tr>
                                    <th><input type="checkbox" id="selectAllItems"></th>
                                    <th>Product</th>
                                    <th>Serial No:</th>
                                    <th>Quantity</th>
                                    <th>Unit Price</th>
                                    <th>Restock Qty</th>
                                    <th>Condition</th>
                                    <th>Total</th>
                                    <th>Discount</th>
                                    <th>POS Price</th>
                                </tr>
                            </thead>
                            <tbody>
                                <!-- Items will be populated dynamically -->
                            </tbody>
                            <tfoot>
                                <tr>
                                    <td colspan="9" class="text-end">Total Refund Amount</td>
                                    <td>
                                        <input type="text" class="form-control" id="totalRefundAmount" value="0.00" readonly style="background: #f8f9fa; border: 1px solid #dee2e6; font-weight: bold; color: #dc3545; text-align: right;"/>
                                    </td>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                    
                    <!-- Notification Options -->
                    <div class="row mt-3 mb-3">
                        <div class="col-12">
                            <div class="d-flex justify-content-between align-items-center">
                                <div class="d-flex align-items-center gap-4">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="sendWhatsAppCheckbox" name="send_whatsapp" value="1" <?php echo ($settings['enable_whatsapp'] ?? '0') === '1' ? 'checked' : ''; ?> <?php echo ($settings['enable_whatsapp'] ?? '0') !== '1' ? 'disabled' : ''; ?>>
                                        <label class="form-check-label" for="sendWhatsAppCheckbox">
                                            <i class="fab fa-whatsapp text-success"></i> WhatsApp
                                            <?php if (($settings['enable_whatsapp'] ?? '0') !== '1'): ?>
                                                <small class="text-muted">(Disabled)</small>
                                            <?php endif; ?>
                                        </label>
                                    </div>
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="sendSmsCheckbox" name="send_sms" value="1" <?php echo ($settings['enable_sms'] ?? '0') === '1' ? 'checked' : ''; ?> <?php echo ($settings['enable_sms'] ?? '0') !== '1' ? 'disabled' : ''; ?>>
                                        <label class="form-check-label" for="sendSmsCheckbox">
                                            <i class="fas fa-sms text-primary"></i> SMS
                                            <?php if (($settings['enable_sms'] ?? '0') !== '1'): ?>
                                                <small class="text-muted">(Disabled)</small>
                                            <?php endif; ?>
                                        </label>
                                    </div>
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="sendEmailCheckbox" name="send_email" value="1" <?php echo ($settings['enable_email'] ?? '0') === '1' ? 'checked' : ''; ?> <?php echo ($settings['enable_email'] ?? '0') !== '1' ? 'disabled' : ''; ?>>
                                        <label class="form-check-label" for="sendEmailCheckbox">
                                            <i class="fas fa-envelope text-info"></i> Email
                                            <?php if (($settings['enable_email'] ?? '0') !== '1'): ?>
                                                <small class="text-muted">(Disabled)</small>
                                            <?php endif; ?>
                                        </label>
                                    </div>
                                </div>
                                <div class="d-flex gap-2">
                                    <button type="button" class="btn btn-outline-info btn-sm" id="previewNotificationBtn">
                                        <i class="fas fa-eye"></i> Preview Message
                                    </button>
                                </div>
                            </div>
                            <small class="text-muted mt-2 d-block">
                                <i class="fas fa-info-circle"></i> Disabled options indicate notification methods not enabled in store settings.
                            </small>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-danger" id="confirmRefundBtn">Process Return</button>
            </div>
        </div>
    </div>
</div>

<!-- Refund Details Modal -->
<div class="modal fade" id="refundDetailsModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Refund Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div id="refundDetailsContent"></div>
            </div>
        </div>
    </div>
</div>

<!-- Credit Details Modal -->
<div class="modal fade" id="creditDetailsModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Credit Sale Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div id="creditDetailsContent"></div>
            </div>
        </div>
    </div>
</div>

<!-- Already Returned Modal -->
<div class="modal fade" id="alreadyReturnedModal" tabindex="-1" aria-labelledby="alreadyReturnedModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="alreadyReturnedModalLabel">
                    <i class="fas fa-exclamation-triangle text-warning"></i> Already Returned
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="text-center">
                    <i class="fas fa-info-circle text-info" style="font-size: 3rem; margin-bottom: 1rem;"></i>
                    <h5>Cannot Process Return</h5>
                    <p class="text-muted">This sale has already been returned and cannot be processed again.</p>
                    <p class="text-muted">If you need to view return details, please check the Refunds tab.</p>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

<!-- Serial Number Selection Modal -->
<div class="modal fade" id="serialNumberModal" tabindex="-1" aria-labelledby="serialNumberModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="serialNumberModalLabel">
                    <i class="fas fa-barcode"></i> Select Serial Numbers to Return
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div id="serialNumberContent">
                    <!-- Content will be populated dynamically -->
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="confirmSerialSelection">Confirm Selection</button>
            </div>
        </div>
    </div>
</div>

<!-- Notification Preview Modal -->
<div class="modal fade" id="notificationPreviewModal" tabindex="-1" aria-labelledby="notificationPreviewModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="notificationPreviewModalLabel">
                    <i class="fas fa-eye"></i> Notification Preview
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="row">
                    <div class="col-md-4">
                        <div class="card">
                            <div class="card-header">
                                <h6 class="mb-0"><i class="fab fa-whatsapp text-success"></i> WhatsApp</h6>
                            </div>
                            <div class="card-body">
                                <div id="whatsappPreview" class="notification-preview"></div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="card">
                            <div class="card-header">
                                <h6 class="mb-0"><i class="fas fa-sms text-primary"></i> SMS</h6>
                            </div>
                            <div class="card-body">
                                <div id="smsPreview" class="notification-preview"></div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="card">
                            <div class="card-header">
                                <h6 class="mb-0"><i class="fas fa-envelope text-info"></i> Email</h6>
                            </div>
                            <div class="card-body">
                                <div id="emailPreview" class="notification-preview"></div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

<!-- Sales Print Section -->
<div class="sales-print-section" style="display:none;">
    <div style="text-align:center; margin-bottom:18px;">
        <div style="font-size:2rem; font-weight:700; color:#222;">POS</div>
        <div style="color:#444; font-size:1.1rem; margin-bottom:2px;">
            <?php echo htmlspecialchars($storeAddress); ?>
        </div>
        <div style="color:#444; font-size:1.1rem; margin-bottom:2px;">
            Phone: <?php echo htmlspecialchars($storePhone); ?>
        </div>
        <div style="color:#444; font-size:1.1rem; margin-bottom:2px;">
            Email: <?php echo htmlspecialchars($storeEmail); ?>
        </div>
    </div>
    <div style="display:flex; justify-content:center; align-items:center; gap:40px; font-size:1.2rem; font-weight:600; margin-bottom:18px;">
        <div>
            <div style="color:#888; font-size:0.95rem; font-weight:400;">Total Sales</div>
            <div style="color:#2563eb; font-size:1.3rem; font-weight:700;">
                <?php echo $currency . ' ' . number_format($stats['total_sales'], 2); ?>
            </div>
        </div>
        <div>
            <div style="color:#888; font-size:0.95rem; font-weight:400;">Products Sold</div>
            <div style="color:#2563eb; font-size:1.3rem; font-weight:700;">
                <?php echo number_format($stats['total_products']); ?>
            </div>
        </div>
        <div>
            <div style="color:#888; font-size:0.95rem; font-weight:400;">Unique Customers</div>
            <div style="color:#2563eb; font-size:1.3rem; font-weight:700;">
                <?php echo number_format($stats['total_customers']); ?>
            </div>
        </div>
        <div>
            <div style="color:#888; font-size:0.95rem; font-weight:400;">Credit Sales</div>
            <div style="color:#2563eb; font-size:1.3rem; font-weight:700;">
                <?php echo $currency . ' ' . number_format($stats['total_credit'], 2); ?>
            </div>
        </div>
    </div>
    <div style="height:3px; background:#222; border-radius:2px; margin-bottom:18px;"></div>
    <div>
        <table style="width:100%; border-collapse:collapse; font-size:1rem;">
            <thead>
                <tr style="background:#f8f9fa;">
                    <th style="padding:8px 6px; border:1px solid #e5e7eb;">Date</th>
                    <th style="padding:8px 6px; border:1px solid #e5e7eb;">Receipt No</th>
                    <th style="padding:8px 6px; border:1px solid #e5e7eb;">Customer</th>
                    <th style="padding:8px 6px; border:1px solid #e5e7eb;">Amount</th>
                    <th style="padding:8px 6px; border:1px solid #e5e7eb;">Payment Method</th>
                    <th style="padding:8px 6px; border:1px solid #e5e7eb;">Status</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($sales as $sale): ?>
                <tr>
                    <td style="padding:8px 6px; border:1px solid #e5e7eb;">
                        <?php echo htmlspecialchars($sale['created_at']); ?>
                    </td>
                    <td style="padding:8px 6px; border:1px solid #e5e7eb;">
                        <?php echo htmlspecialchars($sale['invoice_number'] ?? $sale['id']); ?>
                    </td>
                    <td style="padding:8px 6px; border:1px solid #e5e7eb;">
                        <?php echo htmlspecialchars($sale['customer_name'] ?? 'Walk-in'); ?>
                    </td>
                    <td style="padding:8px 6px; border:1px solid #e5e7eb; text-align:right;">
                        <?php echo $currency . ' ' . number_format($sale['total_amount'], 2); ?>
                    </td>
                    <td style="padding:8px 6px; border:1px solid #e5e7eb;">
                        <?php echo formatPaymentMethods($sale['payment_methods'], $sale['payment_method']); ?>
                    </td>
                    <td style="padding:8px 6px; border:1px solid #e5e7eb;">
                        <?php echo ucfirst($sale['status']); ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>
<!-- Print Footer -->
<div class="sales-print-footer">
    <div class="page-number"></div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script src="https://cdn.datatables.net/1.11.5/js/jquery.dataTables.min.js"></script>
<link href="https://cdn.datatables.net/1.11.5/css/jquery.dataTables.min.css" rel="stylesheet">

<script>
    // Combined Sales & Products Chart
    const combinedCtx = document.getElementById('combinedChart').getContext('2d');
    const combinedChart = new Chart(combinedCtx, {
        type: 'line',
        data: {
            labels: <?php echo json_encode(array_column($daily_sales, 'date')); ?>,
            datasets: [{
                label: 'Daily Sales',
                data: <?php echo json_encode(array_column($daily_sales, 'total')); ?>,
                borderColor: 'rgb(75, 192, 192)',
                backgroundColor: 'rgba(75, 192, 192, 0.1)',
                tension: 0.4,
                fill: true,
                yAxisID: 'y',
                borderWidth: 2
            }, {
                label: 'Top Product Revenue',
                data: <?php echo json_encode(array_column($top_products, 'total_revenue')); ?>,
                borderColor: 'rgb(255, 99, 132)',
                backgroundColor: 'rgba(255, 99, 132, 0.1)',
                tension: 0.4,
                fill: true,
                yAxisID: 'y1',
                borderWidth: 2
            }, {
                label: 'Units Sold',
                data: <?php echo json_encode(array_column($top_products, 'total_quantity')); ?>,
                borderColor: 'rgb(54, 162, 235)',
                backgroundColor: 'rgba(54, 162, 235, 0.1)',
                tension: 0.4,
                fill: true,
                yAxisID: 'y2',
                borderWidth: 2
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            interaction: {
                mode: 'index',
                intersect: false,
            },
            scales: {
                x: {
                    title: {
                        display: false
                    },
                    ticks: {
                        maxRotation: 45,
                        minRotation: 45,
                        font: {
                            size: 10
                        }
                    }
                },
                y: {
                    type: 'linear',
                    display: true,
                    position: 'left',
                    title: {
                        display: false
                    },
                    ticks: {
                        font: {
                            size: 10
                        },
                        callback: function(value) {
                            return '<?php echo $currency; ?> ' + value.toLocaleString();
                        }
                    }
                },
                y1: {
                    type: 'linear',
                    display: true,
                    position: 'right',
                    title: {
                        display: false
                    },
                    grid: {
                        drawOnChartArea: false
                    },
                    ticks: {
                        font: {
                            size: 10
                        },
                        callback: function(value) {
                            return '<?php echo $currency; ?> ' + value.toLocaleString();
                        }
                    }
                },
                y2: {
                    type: 'linear',
                    display: true,
                    position: 'right',
                    title: {
                        display: false
                    },
                    grid: {
                        drawOnChartArea: false
                    },
                    ticks: {
                        font: {
                            size: 10
                        }
                    }
                }
            },
            plugins: {
                legend: {
                    position: 'top',
                    labels: {
                        font: {
                            size: 11
                        },
                        usePointStyle: true,
                        padding: 15
                    }
                },
                tooltip: {
                    backgroundColor: 'rgba(0, 0, 0, 0.8)',
                    titleFont: {
                        size: 12
                    },
                    bodyFont: {
                        size: 11
                    },
                    callbacks: {
                        label: function(context) {
                            let label = context.dataset.label || '';
                            if (label) {
                                label += ': ';
                            }
                            if (context.dataset.yAxisID === 'y' || context.dataset.yAxisID === 'y1') {
                                label += '<?php echo $currency; ?> ' + context.parsed.y.toLocaleString();
                            } else {
                                label += context.parsed.y.toLocaleString();
                            }
                            return label;
                        }
                    }
                }
            }
        }
    });

    // Credit Sales Chart
    const creditCtx = document.getElementById('creditSalesChart').getContext('2d');
    const creditSalesChart = new Chart(creditCtx, {
        type: 'line',
        data: {
            labels: <?php echo json_encode(array_column($daily_sales, 'date')); ?>,
            datasets: [{
                label: 'Credit Sales',
                data: <?php echo json_encode(array_column($daily_sales, 'total')); ?>,
                borderColor: 'rgb(99, 102, 241)',
                backgroundColor: 'rgba(99, 102, 241, 0.1)',
                tension: 0.4,
                fill: true
            }]
        },
        options: {
            responsive: true,
            interaction: {
                mode: 'index',
                intersect: false,
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        callback: function(value) {
                            return '<?php echo $currency; ?> ' + value.toLocaleString();
                        }
                    }
                }
            },
            plugins: {
                legend: {
                    position: 'top',
                },
                tooltip: {
                    callbacks: {
                        label: function(context) {
                            let label = context.dataset.label || '';
                            if (label) {
                                label += ': ';
                            }
                            label += '<?php echo $currency; ?> ' + context.parsed.y.toLocaleString();
                            return label;
                        }
                    }
                }
            }
        }
    });

    // Handle print event
    window.addEventListener('beforeprint', function() {
        combinedChart.resize();
        creditSalesChart.resize();
    });

    // Live search functionality for sales table
    document.getElementById('salesSearch').addEventListener('input', function() {
        const searchTerm = this.value.toLowerCase();
        const tableRows = document.querySelectorAll('#salesTable tbody tr');
        
        tableRows.forEach(row => {
            const customerName = row.querySelector('td:nth-child(3)').textContent.toLowerCase();
            const receiptNumber = row.querySelector('td:nth-child(2)').textContent.toLowerCase();
            const amount = row.querySelector('td:nth-child(4)').textContent.toLowerCase();
            
            const matches = customerName.includes(searchTerm) || 
                           receiptNumber.includes(searchTerm) || 
                           amount.includes(searchTerm);
            
            row.style.display = matches ? '' : 'none';
        });
        
        // Update row count
        const visibleRows = document.querySelectorAll('#salesTable tbody tr:not([style*="display: none"])');
        const totalRows = tableRows.length;
        
        // Show search results count
        const searchResults = document.createElement('div');
        searchResults.className = 'search-results-info';
        searchResults.innerHTML = `Showing ${visibleRows.length} of ${totalRows} sales`;
        
        // Remove existing search results info
        const existingInfo = document.querySelector('.search-results-info');
        if (existingInfo) {
            existingInfo.remove();
        }
        
        // Add search results info if searching
        if (searchTerm) {
            const tableContainer = document.querySelector('#salesTable').parentElement;
            tableContainer.insertBefore(searchResults, document.querySelector('#salesTable'));
        }
    });

    // Refund logic
    let refundMaxAmount = 0;
    let saleItems = [];
    let currentSaleData = null;
    let selectedSerialNumbers = {}; // Store selected serial numbers for each item



    function updateRefundTotals() {
        let total = 0;
        console.log('Updating refund totals...');
        
        document.querySelectorAll('#refundItemsTable tbody tr').forEach(row => {
            const checkbox = row.querySelector('.item-checkbox');
            if (checkbox && checkbox.checked) {
                const condition = row.querySelector('select[name="condition"]').value;
                let itemTotal = 0;
                if (["Damaged", "Opened", "Other"].includes(condition)) {
                    itemTotal = parseFloat(row.querySelector('input[name="manual_total"]').value) || 0;
                } else {
                    const quantity = parseFloat(row.querySelector('input[name="quantity"]').value) || 0;
                    const unitPrice = parseFloat(row.querySelector('input[name="unit_price"]').value) || 0;
                    const discount = parseFloat(row.querySelector('.discount-amount').textContent) || 0;
                    // Calculate total after discount
                    itemTotal = (quantity * unitPrice) - discount;
                }
                row.querySelector('.item-total').textContent = itemTotal.toFixed(2);
                total += itemTotal;
                console.log('Item total:', itemTotal.toFixed(2), 'Running total:', total.toFixed(2));
            }
        });
        
        const totalElement = document.getElementById('totalRefundAmount');
        if (totalElement) {
            totalElement.value = total.toFixed(2);
            console.log('Final total set to:', total.toFixed(2));
        } else {
            console.error('Total refund amount element not found!');
        }
    }

    document.getElementById('confirmRefundBtn').addEventListener('click', function() {
        const saleId = document.getElementById('refundSaleId').value;
        const confirmBtn = this;
        const items = [];
        let hasSelectedItems = false;
        let totalRefundAmount = 0;

        // Get the original sale amount
        const saleRow = document.querySelector(`tr[data-sale-id="${saleId}"]`);
        if (!saleRow) {
            alert('Could not find sale details.');
            return;
        }

        const saleAmount = parseFloat(saleRow.querySelector('td:nth-child(4)').textContent.replace(/[^\d.]/g, ''));

        // Get all refund items
        document.querySelectorAll('#refundItemsTable tbody tr').forEach(row => {
            const checkbox = row.querySelector('.item-checkbox');
            if (checkbox && checkbox.checked) {
                hasSelectedItems = true;
                const condition = row.querySelector('select[name="condition"]').value;
                const quantity = parseInt(row.querySelector('input[name="quantity"]').value) || 0;
                const restockQuantity = parseInt(row.querySelector('input[name="restock_quantity"]').value) || 0;
                const unitPrice = parseFloat(row.querySelector('input[name="unit_price"]').value) || 0;
                const discount = parseFloat(row.querySelector('.discount-amount').textContent) || 0;
                let totalPrice = 0;
                if (["Damaged", "Opened", "Other"].includes(condition)) {
                    totalPrice = parseFloat(row.querySelector('input[name="manual_total"]').value) || 0;
                } else {
                    totalPrice = (quantity * unitPrice) - discount;
                }

                if (quantity <= 0 || unitPrice <= 0) {
                    alert('Invalid quantity or unit price for one or more items.');
                    return;
                }

                totalRefundAmount += totalPrice;

                items.push({
                    sale_item_id: row.dataset.saleItemId,
                    product_id: row.dataset.productId,
                    quantity: quantity,
                    restock_quantity: restockQuantity,
                    unit_price: unitPrice,
                    total_price: totalPrice,
                    discount: discount,
                    condition: condition,
                    selected_serials: selectedSerialNumbers[row.dataset.saleItemId] || []
                });
            }
        });

        if (!hasSelectedItems) {
            alert('Please select at least one item to refund.');
            return;
        }

        // Validate refund amount based on payment method
        if (window.currentSaleCreditInfo && window.currentSaleCreditInfo.is_credit_sale) {
            // Check if all items being returned are in "Good" condition
            const allItemsGood = items.every(item => item.condition === 'Good');
            
            if (allItemsGood) {
                // For "Good" condition items, allow return regardless of credit balance (including 0.00)
                // The refund will create a credit balance or reduce existing balance
                console.log('All items are in Good condition - allowing return regardless of credit balance');
            } else {
                // For damaged/opened items, check credit balance
                if (totalRefundAmount > window.currentSaleCreditInfo.credit_balance) {
                    alert(`Refund amount (${totalRefundAmount.toFixed(2)}) cannot exceed credit balance (${window.currentSaleCreditInfo.credit_balance.toFixed(2)}) for damaged/opened items.`);
                    return;
                }
            }
        } else {
            if (totalRefundAmount > saleAmount) {
                alert(`Refund amount (${totalRefundAmount.toFixed(2)}) cannot exceed sale total (${saleAmount.toFixed(2)})`);
                return;
            }
        }

        // Confirm refund
        if (!confirm('Are you sure you want to process this return? This action cannot be undone.')) {
            return;
        }

        // Show loading state
        confirmBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';
        confirmBtn.disabled = true;

        // Get notification preferences
        const sendWhatsApp = document.getElementById('sendWhatsAppCheckbox').checked;
        const sendSMS = document.getElementById('sendSmsCheckbox').checked;
        const sendEmail = document.getElementById('sendEmailCheckbox').checked;
        
        // Process refund
        fetch('../api/process_refund.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                sale_id: saleId,
                refund_amount: totalRefundAmount,
                refund_reason: 'Customer Return',
                refund_items: items.map(item => ({
                    product_id: item.product_id,
                    quantity: parseInt(item.quantity),
                    price: parseFloat(item.unit_price),
                    discount: parseFloat(item.discount || 0),
                    condition: item.condition,
                    restock_quantity: parseInt(item.restock_quantity),
                    selected_serials: item.selected_serials
                })),
                notifications: {
                    send_whatsapp: sendWhatsApp,
                    send_sms: sendSMS,
                    send_email: sendEmail
                }
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Show success message
                const alertDiv = document.createElement('div');
                alertDiv.className = 'alert alert-success alert-dismissible fade show';
                
                let successMessage = '<strong>Success!</strong> Return processed successfully.';
                
                if (data.has_multiple_payments && data.refund_breakdown) {
                    successMessage += '<br><br><strong>Refund Breakdown:</strong>';
                    data.refund_breakdown.forEach(refund => {
                        successMessage += `<br>${refund.method.charAt(0).toUpperCase() + refund.method.slice(1)}: ₱${refund.refund_amount.toFixed(2)}`;
                    });
                } else if (data.is_credit_sale) {
                    successMessage += '<br><strong>Credit Balance:</strong> The refund amount has been applied to clear the credit balance.';
                } else {
                    successMessage += '<br><strong>Payment Method:</strong> ' + data.payment_method + ' refund processed.';
                }
                
                if (data.restocked_items > 0) {
                    successMessage += `<br><strong>Stock:</strong> ${data.restocked_items} items have been returned to stock.`;
                }
                
                if (data.is_complete_refund) {
                    successMessage += '<br><strong>Status:</strong> Sale status updated to "Refunded" (all items returned).';
                } else {
                    successMessage += '<br><strong>Status:</strong> Partial return processed. Sale remains active for remaining items.';
                }
                
                // Add notification results
                if (data.notifications_sent && Object.keys(data.notifications_sent).length > 0) {
                    successMessage += '<br><br><strong>Notifications Sent:</strong>';
                    
                    if (data.notifications_sent.sms) {
                        const smsStatus = data.notifications_sent.sms.success ? '✅ SMS sent' : '❌ SMS failed';
                        const smsError = data.notifications_sent.sms.error ? ` (${data.notifications_sent.sms.error})` : '';
                        successMessage += '<br>' + smsStatus + smsError;
                    }
                    
                    if (data.notifications_sent.email) {
                        const emailStatus = data.notifications_sent.email.success ? '✅ Email sent' : '❌ Email failed';
                        const emailError = data.notifications_sent.email.error ? ` (${data.notifications_sent.email.error})` : '';
                        successMessage += '<br>' + emailStatus + emailError;
                    }
                    
                    if (data.notifications_sent.whatsapp) {
                        const whatsappStatus = data.notifications_sent.whatsapp.success ? '✅ WhatsApp sent' : '❌ WhatsApp failed';
                        const whatsappError = data.notifications_sent.whatsapp.error ? ` (${data.notifications_sent.whatsapp.error})` : '';
                        successMessage += '<br>' + whatsappStatus + whatsappError;
                        
                        // Auto-open WhatsApp link if available
                        if (data.notifications_sent.whatsapp.success && data.notifications_sent.whatsapp.auto_open_url) {
                            successMessage += '<br><small class="text-info">📱 WhatsApp link will open automatically in a new tab</small>';
                            setTimeout(() => {
                                window.open(data.notifications_sent.whatsapp.auto_open_url, '_blank');
                            }, 1000);
                        }
                    }
                }
                
                alertDiv.innerHTML = `
                    ${successMessage}
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                `;
                const container = document.querySelector('.container-fluid');
                if (container) {
                    const firstCard = container.querySelector('.card');
                    if (firstCard && firstCard.parentNode === container) {
                        container.insertBefore(alertDiv, firstCard);
                    } else {
                        container.appendChild(alertDiv);
                    }
                } else {
                    document.body.appendChild(alertDiv);
                }

                // Close modal and clear credit info
                var modal = bootstrap.Modal.getInstance(document.getElementById('refundModal'));
                modal.hide();
                
                // Clear credit info and any credit info div
                window.currentSaleCreditInfo = null;
                const creditInfoDiv = document.querySelector('#refundModal .alert-info');
                if (creditInfoDiv) {
                    creditInfoDiv.remove();
                }

                // Open refund receipt in new window
                if (data.refund_id) {
                    const receiptWindow = window.open(`../receipts/refund_receipt.php?refund_id=${data.refund_id}&print=1`, '_blank');
                    if (receiptWindow) {
                        receiptWindow.focus();
                    }
                }

                // Refresh the page after a short delay
                setTimeout(() => {
                    window.location.reload();
                }, 2000);
            } else {
                alert('Refund failed: ' + (data.message || 'Unknown error'));
            }
        })
        .catch(err => {
            console.error('Refund error:', err);
            alert('Refund error: ' + err);
        })
        .finally(() => {
            confirmBtn.innerHTML = 'Process Return';
            confirmBtn.disabled = false;
        });
    });

    // Add function to reprint refund receipt
    function reprintRefundReceipt(refundId) {
        window.open(`refund_receipt.php?refund_id=${refundId}&print=1`, '_blank');
    }
    

    
    // Preview notification message
    function previewNotificationMessage() {
        const saleId = document.getElementById('refundSaleId').value;
        const selectedItems = [];
        
        // Get selected items
        document.querySelectorAll('#refundItemsTable tbody tr').forEach(row => {
            const checkbox = row.querySelector('.item-checkbox');
            if (checkbox && checkbox.checked) {
                const quantity = parseInt(row.querySelector('input[name="quantity"]').value) || 0;
                const condition = row.querySelector('select[name="condition"]').value;
                const productName = row.querySelector('td:nth-child(2)').textContent.trim();
                
                selectedItems.push({
                    quantity: quantity,
                    condition: condition,
                    name: productName
                });
            }
        });
        
        if (selectedItems.length === 0) {
            alert('Please select at least one item to return.');
            return;
        }
        
        // Calculate total refund amount
        let totalRefundAmount = 0;
        document.querySelectorAll('#refundItemsTable tbody tr').forEach(row => {
            const checkbox = row.querySelector('.item-checkbox');
            if (checkbox && checkbox.checked) {
                const quantity = parseInt(row.querySelector('input[name="quantity"]').value) || 0;
                const unitPrice = parseFloat(row.querySelector('input[name="unit_price"]').value) || 0;
                const discount = parseFloat(row.querySelector('.discount-amount').textContent) || 0;
                const condition = row.querySelector('select[name="condition"]').value;
                
                let itemTotal = 0;
                if (["Damaged", "Opened", "Other"].includes(condition)) {
                    itemTotal = parseFloat(row.querySelector('input[name="manual_total"]').value) || 0;
                } else {
                    itemTotal = (quantity * unitPrice) - discount;
                }
                
                totalRefundAmount += itemTotal;
            }
        });
        
        // Generate message
        const itemsList = selectedItems.map(item => 
            `${item.quantity}x ${item.name} (${item.condition})`
        ).join(', ');
        
        const message = `SALES RETURN CONFIRMATION - <?php echo htmlspecialchars($storeName); ?>.

Dear ${currentSaleData?.customer_name || 'Customer'},
We have received your Sales return:

Return Details:
Receipt: INV-${saleId.padStart(6, '0')}
Items: ${itemsList}
Date: ${new Date().toLocaleDateString('en-US', { 
    year: 'numeric', 
    month: 'short', 
    day: 'numeric',
    hour: '2-digit',
    minute: '2-digit'
})}

Thank you for choosing <?php echo htmlspecialchars($storeName); ?>!
For any questions, please contact us | <?php echo htmlspecialchars($storePhone); ?>`;
        
        // Update preview modals
        const whatsappPreview = document.getElementById('whatsappPreview');
        const smsPreview = document.getElementById('smsPreview');
        const emailPreview = document.getElementById('emailPreview');
        
        if (whatsappPreview) {
            // Create WhatsApp link
            const whatsappMessage = encodeURIComponent(message);
            const whatsappLink = `https://wa.me/?text=${whatsappMessage}`;
            whatsappPreview.innerHTML = `
                <div class="mb-2">
                    <strong>Message Preview:</strong>
                </div>
                <div class="mb-3" style="white-space: pre-wrap; font-family: monospace; font-size: 11px;">${message}</div>
                <a href="${whatsappLink}" target="_blank" class="btn btn-success btn-sm">
                    <i class="fab fa-whatsapp"></i> Open in WhatsApp
                </a>
            `;
        }
        
        if (smsPreview) smsPreview.textContent = message;
        if (emailPreview) emailPreview.textContent = message;
        
        // Show preview modal
        const previewModal = new bootstrap.Modal(document.getElementById('notificationPreviewModal'));
        previewModal.show();
    }

    // Function to show refund modal
    function showRefundModal(saleId) {
        // Clear selected serial numbers for new modal
        selectedSerialNumbers = {};
        
        // Find the sale row and get the amount
        const row = document.querySelector(`tr[data-sale-id="${saleId}"]`);
        if (!row) {
            alert('Could not find sale details.');
            return;
        }
        
        refundMaxAmount = parseFloat(row.querySelector('td:nth-child(4)').textContent.replace(/[^\d.]/g, ''));
        document.getElementById('refundSaleId').value = saleId;
        
        // Notification settings are now handled by PHP directly
        
        // Fetch sale items
        fetch('../api/get_sale_items.php?sale_id=' + saleId)
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    saleItems = data.items;
                    currentSaleData = data;
                    
                    // Debug: Show data received
                    console.log('=== MODAL DATA ===');
                    console.log('Total items:', saleItems.length);
                    console.log('First item sample:', saleItems[0]);
                    console.log('==================');
                    
                    const tbody = document.querySelector('#refundItemsTable tbody');
                    tbody.innerHTML = '';
                    
                    // Check if any items can be returned
                    const canReturnAny = data.can_return_any;
                    if (!canReturnAny) {
                        alert('All items in this sale are older than 7 days and cannot be returned.');
                        return;
                    }
                    
                    saleItems.forEach(item => {
                        // Debug logging
                        console.log('=== ITEM DEBUG ===');
                        console.log('Item:', item.name);
                        console.log('is_returned:', item.is_returned);
                        console.log('can_return:', item.can_return);
                        console.log('returned_quantity:', item.returned_quantity);
                        console.log('quantity:', item.quantity);
                        console.log('remaining_quantity:', item.remaining_quantity);
                        console.log('==================');
                        
                        const tr = document.createElement('tr');
                        tr.dataset.saleItemId = item.id;
                        tr.dataset.productId = item.product_id;
                        
                        // Add styling for returned items and grey out items that can't be returned
                        if (item.is_returned) {
                            tr.classList.add('item-returned');
                            tr.classList.add('item-disabled');
                            tr.style.opacity = '0.6';
                        }
                        if (!item.can_return) {
                            tr.classList.add('item-disabled');
                            tr.style.opacity = '0.6';
                        }
                        
                        // Format serial numbers for display
                        let serialNumbersDisplay = '';
                        if (item.serial_numbers && item.serial_numbers.length > 0) {
                            const serials = item.serial_numbers.map(sn => sn.serial_number);
                            serialNumbersDisplay = serials.join(', ');
                        } else {
                            serialNumbersDisplay = 'No Serial';
                        }
                        
                        // Add return policy indicator
                        const returnPolicyClass = item.can_return ? 'text-success' : 'text-danger';
                        const returnPolicyIcon = item.can_return ? 'fa-check-circle' : 'fa-times-circle';
                        
                        const checkboxDisabled = (!item.can_return || item.is_returned);
                        console.log('Checkbox disabled for', item.name, ':', checkboxDisabled, '(!item.can_return || item.is_returned) =', '(!' + item.can_return + ' || ' + item.is_returned + ')');
                        
                        tr.innerHTML = `
                            <td>
                                <input type="checkbox" class="item-checkbox" ${checkboxDisabled ? 'disabled' : ''} 
                                       ${!item.can_return ? 'title="This item cannot be returned (older than 7 days)"' : ''}
                                       ${item.is_returned ? 'title="This item has already been returned"' : ''}>
                            </td>
                            <td>
                                ${item.name}
                                <br><small class="${returnPolicyClass}">
                                    <i class="fas ${returnPolicyIcon}"></i> ${item.return_policy_message}
                                </small>
                                ${item.returned_quantity > 0 ? `<br><small class="text-info">Returned: ${item.returned_quantity} of ${item.quantity}</small>` : ''}
                            </td>
                            <td style="font-size: 0.85em; color: #495057; word-break: break-all; font-family: monospace; background-color: #f8f9fa; padding: 4px 6px; border-radius: 3px;">
                                ${serialNumbersDisplay}
                                ${item.has_serials && item.available_serials.length > 0 && item.can_return ? 
                                    `<br><button type="button" class="btn btn-sm btn-outline-primary mt-1" onclick="selectSerialNumbers(${item.id}, '${item.name}', ${JSON.stringify(item.available_serials)})">
                                        <i class="fas fa-barcode"></i> Select Serial Numbers
                                    </button>` : ''
                                }
                            </td>
                            <td>
                                <input type="number" name="quantity" class="form-control form-control-sm" 
                                       min="1" max="${item.remaining_quantity}" value="${item.remaining_quantity}" 
                                       style="width: 80px;" ${(!item.can_return || item.is_returned) ? 'disabled' : ''}>
                            </td>
                            <td>
                                <input type="number" name="unit_price" class="form-control form-control-sm" 
                                       value="${item.unit_price}" readonly style="width: 100px;">
                            </td>
                            <td>
                                <input type="number" name="restock_quantity" class="form-control form-control-sm" 
                                       min="0" max="${item.remaining_quantity}" value="${item.remaining_quantity}" 
                                       style="width: 80px;" ${(!item.can_return || item.is_returned) ? 'disabled' : ''}>
                            </td>
                            <td>
                                <select name="condition" class="form-select form-select-sm" style="width: 120px;" ${(!item.can_return || item.is_returned) ? 'disabled' : ''}>
                                    <option value="Good">Good</option>
                                    <option value="Damaged">Damaged</option>
                                    <option value="Opened">Opened</option>
                                    <option value="Other">Other</option>
                                </select>
                            </td>
                            <td class="item-total-cell">
                                <input type="number" name="manual_total" class="form-control form-control-sm d-none" min="0" max="${item.remaining_quantity * item.unit_price}" step="0.01" style="width: 100px;">
                                <span class="item-total">${(item.remaining_quantity * item.unit_price).toFixed(2)}</span>
                            </td>
                            <td>
                                <span class="discount-amount">${item.discount.toFixed(2)}</span>
                            </td>
                            <td>
                                <span class="pos-price">${(item.unit_price - item.discount).toFixed(2)}</span>
                            </td>
                        `;
                        tbody.appendChild(tr);

                        // Attach event listeners for this row
                        const conditionSelect = tr.querySelector('select[name="condition"]');
                        const manualTotalInput = tr.querySelector('input[name="manual_total"]');
                        const itemTotalSpan = tr.querySelector('.item-total');
                        const quantityInput = tr.querySelector('input[name="quantity"]');
                        const unitPriceInput = tr.querySelector('input[name="unit_price"]');
                        const restockInput = tr.querySelector('input[name="restock_quantity"]');
                        const checkbox = tr.querySelector('.item-checkbox');

                        function updateTotalField() {
                            const condition = conditionSelect.value;
                            const qty = parseFloat(quantityInput.value) || 0;
                            const price = parseFloat(unitPriceInput.value) || 0;
                            const maxTotal = qty * price;
                            manualTotalInput.max = maxTotal;
                            if (["Damaged", "Opened", "Other"].includes(condition)) {
                                manualTotalInput.classList.remove('d-none');
                                manualTotalInput.value = itemTotalSpan.textContent;
                                itemTotalSpan.classList.add('d-none');
                                // Set restock qty to 0 and disable
                                restockInput.value = 0;
                                restockInput.disabled = true;
                            } else {
                                manualTotalInput.classList.add('d-none');
                                itemTotalSpan.classList.remove('d-none');
                                itemTotalSpan.textContent = (qty * price).toFixed(2);
                                // Set restock qty to match refund qty and enable
                                restockInput.value = qty;
                                restockInput.disabled = false;
                            }
                            updateRefundTotals();
                        }
                        
                        // Real-time validation for manual total
                        manualTotalInput.addEventListener('input', function() {
                            const qty = parseFloat(quantityInput.value) || 0;
                            const price = parseFloat(unitPriceInput.value) || 0;
                            const maxTotal = qty * price;
                            if (parseFloat(this.value) > maxTotal) {
                                this.value = maxTotal.toFixed(2);
                            }
                            updateRefundTotals();
                        });
                        
                        // Initial state
                        updateTotalField();
                        
                        // Event listeners for this row
                        conditionSelect.addEventListener('change', updateTotalField);
                        quantityInput.addEventListener('input', updateTotalField);
                        manualTotalInput.addEventListener('input', updateRefundTotals);
                        checkbox.addEventListener('change', updateRefundTotals);
                    });
                    
                    // Select all checkbox - only select eligible items
                    const selectAllCheckbox = document.getElementById('selectAllItems');
                    selectAllCheckbox.addEventListener('change', function() {
                        document.querySelectorAll('.item-checkbox:not(:disabled)').forEach(checkbox => {
                            checkbox.checked = this.checked;
                        });
                        updateRefundTotals();
                    });
                    
                    // Reset select all checkbox state
                    selectAllCheckbox.checked = false;
                    selectAllCheckbox.indeterminate = false;

                    // Update totals initially
                    updateRefundTotals();
                    
                    // Show the modal
                    var modal = new bootstrap.Modal(document.getElementById('refundModal'));
                    modal.show();
                } else {
                    alert('Error loading sale items: ' + (data.message || 'Unknown error'));
                }
            })
            .catch(err => {
                console.error('Error fetching sale items:', err);
                alert('Error loading sale items: ' + err);
            });
    }

    // Initialize all DataTables
    $(document).ready(function() {
        // Clear credit info when refund modal is closed
        $('#refundModal').on('hidden.bs.modal', function () {
            window.currentSaleCreditInfo = null;
            selectedSerialNumbers = {};
            const creditInfoDiv = document.querySelector('#refundModal .alert-info');
            if (creditInfoDiv) {
                creditInfoDiv.remove();
            }
        });
        
        // Destroy existing DataTable instance if it exists
        if ($.fn.DataTable.isDataTable('#refundsTable')) {
            $('#refundsTable').DataTable().destroy();
        }

        // Initialize refunds table
        var refundsTable = $('#refundsTable').DataTable({
            order: [[0, 'desc']],
            pageLength: 25,
            dom: '<"row"<"col-sm-12 col-md-6"l><"col-sm-12 col-md-6"f>>rtip',
            language: {
                search: "",
                searchPlaceholder: "Search by customer name or refund ID..."
            },
            initComplete: function() {
                // Hide the default search box
                $('.dataTables_filter').hide();
            }
        });

        // Custom search functionality
        $('#refundSearch').on('keyup', function() {
            var searchValue = this.value.toLowerCase();
            
            // Custom filtering function
            $.fn.dataTable.ext.search = [];
            $.fn.dataTable.ext.search.push(
                function(settings, data, dataIndex) {
                    var customerName = data[3].toLowerCase();
                    var refundId = data[1].toLowerCase();
                    
                    if (customerName.indexOf(searchValue) !== -1 || refundId.indexOf(searchValue) !== -1) {
                        return true;
                    }
                    return false;
                }
            );
            
            refundsTable.draw();
        });

        // After creating all rows in the refund modal:
        document.querySelectorAll('.item-checkbox').forEach(cb => {
            cb.addEventListener('change', updateRefundTotals);
        });
    });

    // Add this to your existing JavaScript
    function showRefundDetails(refundId) {
        // Show loading state
        const modalBody = document.querySelector('#refundDetailsModal .modal-body');
        modalBody.innerHTML = '<div class="text-center"><div class="spinner-border text-primary" role="status"></div><p class="mt-2">Loading refund details...</p></div>';
        
        // Show the modal
        const modal = new bootstrap.Modal(document.getElementById('refundDetailsModal'));
        modal.show();

        // Fetch refund details
        fetch(`../api/get_refund_details.php?refund_id=${refundId}`)
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    const refund = data.refund;
                    const items = data.items;
                    
                    // Create the HTML content
                    let html = `
                        <div class="refund-details">
                            <div class="row mb-4">
                                <div class="col-md-6">
                                    <h5>Refund Information</h5>
                                    <table class="table table-sm">
                                        <tr>
                                            <th>Refund ID:</th>
                                            <td>${refund.refund_id}</td>
                                        </tr>
                                        <tr>
                                            <th>Date:</th>
                                            <td>${refund.refund_date}</td>
                                        </tr>
                                        <tr>
                                            <th>Original Sale ID:</th>
                                            <td>${refund.sale_id}</td>
                                        </tr>
                                        <tr>
                                            <th>Amount:</th>
                                            <td>₱${parseFloat(refund.amount).toFixed(2)}</td>
                                        </tr>
                                        <tr>
                                            <th>Reason:</th>
                                            <td>${refund.reason}</td>
                                        </tr>
                                    </table>
                                </div>
                            </div>
                            
                            <h5>Refunded Items</h5>
                            <div class="table-responsive">
                                <table class="table table-sm">
                                    <thead>
                                        <tr>
                                            <th>Item</th>
                                            <th>Quantity</th>
                                            <th>Unit Price</th>
                                            <th>Condition</th>
                                            <th>Restock Qty</th>
                                            <th>Total</th>
                                        </tr>
                                    </thead>
                                    <tbody>`;

                    items.forEach(item => {
                        html += `
                            <tr>
                                <td>${item.item_name}</td>
                                <td>${item.quantity}</td>
                                <td>₱${parseFloat(item.unit_price).toFixed(2)}</td>
                                <td>${item.condition}</td>
                                <td>${item.restock_quantity}</td>
                                <td>₱${parseFloat(item.total).toFixed(2)}</td>
                            </tr>`;
                    });

                    html += `
                                    </tbody>
                                </table>
                            </div>
                        </div>`;

                    modalBody.innerHTML = html;
                } else {
                    modalBody.innerHTML = `<div class="alert alert-danger">${data.message}</div>`;
                }
            })
            .catch(error => {
                console.error('Error:', error);
                modalBody.innerHTML = '<div class="alert alert-danger">Error loading refund details</div>';
            });
    }

    // Function to get payment method icon
    function getPaymentMethodIcon(paymentMethod) {
        switch(paymentMethod.toLowerCase()) {
            case 'cash':
                return '<i class="fas fa-money-bill-wave text-success"></i>';
            case 'mpesa':
                return '<i class="fas fa-mobile-alt text-primary"></i>';
            case 'credit':
                return '<i class="fas fa-credit-card text-warning"></i>';
            default:
                return '<i class="fas fa-money-bill-wave text-success"></i>';
        }
    }

    // Function to show already refunded modal
    function showAlreadyReturnedModal() {
        const modal = new bootstrap.Modal(document.getElementById('alreadyReturnedModal'));
        modal.show();
    }

    // Function to show serial number selection modal
    function selectSerialNumbers(itemId, itemName, availableSerials) {
        const modal = document.getElementById('serialNumberModal');
        const content = document.getElementById('serialNumberContent');
        
        // Store current item ID for reference
        window.currentSerialSelectionItemId = itemId;
        
        // Initialize selected serials for this item if not exists
        if (!selectedSerialNumbers[itemId]) {
            selectedSerialNumbers[itemId] = [];
        }
        
        content.innerHTML = `
            <div class="mb-3">
                <h6>${itemName}</h6>
                <p class="text-muted">Select which serial numbers to return:</p>
            </div>
            <div class="row">
                ${availableSerials.map(serial => `
                    <div class="col-md-4 mb-2">
                        <div class="form-check">
                            <input class="form-check-input serial-checkbox" type="checkbox" 
                                   value="${serial}" id="serial_${itemId}_${serial}" 
                                   ${selectedSerialNumbers[itemId].includes(serial) ? 'checked' : ''}>
                            <label class="form-check-label" for="serial_${itemId}_${serial}">
                                ${serial}
                            </label>
                        </div>
                    </div>
                `).join('')}
            </div>
            <div class="mt-3">
                <button type="button" class="btn btn-sm btn-outline-primary me-2" onclick="selectAllSerials()">
                    <i class="fas fa-check-double"></i> Select All
                </button>
                <button type="button" class="btn btn-sm btn-outline-secondary" onclick="clearAllSerials()">
                    <i class="fas fa-times"></i> Clear All
                </button>
            </div>
        `;
        
        // Show the modal
        const bootstrapModal = new bootstrap.Modal(modal);
        bootstrapModal.show();
    }

    // Function to select all serial numbers
    function selectAllSerials() {
        document.querySelectorAll('.serial-checkbox').forEach(checkbox => {
            checkbox.checked = true;
        });
    }

    // Function to clear all serial numbers
    function clearAllSerials() {
        document.querySelectorAll('.serial-checkbox').forEach(checkbox => {
            checkbox.checked = false;
        });
    }

    // Function to confirm serial number selection
    document.getElementById('confirmSerialSelection').addEventListener('click', function() {
        const selectedSerials = [];
        document.querySelectorAll('.serial-checkbox:checked').forEach(checkbox => {
            selectedSerials.push(checkbox.value);
        });
        
        // Use the stored item ID from the modal
        const itemId = window.currentSerialSelectionItemId;
        if (itemId) {
            selectedSerialNumbers[itemId] = selectedSerials;
            
            // Update the display in the refund modal to show selected serials
            const itemRow = document.querySelector(`tr[data-sale-item-id="${itemId}"]`);
            if (itemRow) {
                const serialCell = itemRow.querySelector('td:nth-child(3)');
                if (serialCell) {
                    const originalSerials = serialCell.querySelector('.original-serials') || serialCell;
                    const selectedDisplay = selectedSerials.length > 0 ? 
                        `<br><small class="text-success"><strong>Selected: ${selectedSerials.join(', ')}</strong></small>` : '';
                    
                    // Update the serial number display
                    const serialDisplay = originalSerials.innerHTML.replace(/<br><small class="text-success"><strong>Selected:.*?<\/strong><\/small>/g, '');
                    serialCell.innerHTML = serialDisplay + selectedDisplay;
                }
            }
        }
        
        // Close the modal
        const modal = bootstrap.Modal.getInstance(document.getElementById('serialNumberModal'));
        modal.hide();
    });

    // Add this function to handle the Return button click
    function showRefundModal(saleId) {
        console.log('Opening refund modal for sale:', saleId);
        document.getElementById('refundSaleId').value = saleId;
        
        // Reset total refund amount
        document.getElementById('totalRefundAmount').value = '0.00';
        
        // Show loading state in modal
        const tbody = document.querySelector('#refundItemsTable tbody');
        tbody.innerHTML = '<tr><td colspan="10" class="text-center"><i class="fas fa-spinner fa-spin"></i> Loading sale items...</td></tr>';
        
        // Show the modal first
        const modal = new bootstrap.Modal(document.getElementById('refundModal'));
        modal.show();
        
        // First, check credit balance for this sale
        fetch('../api/check_sale_credit_balance.php?sale_id=' + saleId)
            .then(res => res.json())
            .then(creditData => {
                console.log('Credit balance data:', creditData);
                
                // Store credit information globally
                window.currentSaleCreditInfo = creditData;
                
                // Now fetch sale items
                return fetch('../api/get_sale_items.php?sale_id=' + saleId);
            })
            .then(res => {
                if (!res.ok) {
                    throw new Error('Network response was not ok');
                }
                return res.json();
            })
            .then(data => {
                console.log('Received sale items:', data);
                if (data.success) {
                    saleItems = data.items;
                    
                    // Set refund max amount based on payment method
                    if (window.currentSaleCreditInfo && window.currentSaleCreditInfo.is_credit_sale) {
                        refundMaxAmount = window.currentSaleCreditInfo.credit_balance;
                        // Show credit balance info
                        const creditInfoDiv = document.createElement('div');
                        creditInfoDiv.className = 'alert alert-info mb-3';
                        creditInfoDiv.innerHTML = `
                            <strong>Credit Sale Information:</strong><br>
                            Customer: ${window.currentSaleCreditInfo.customer_name}<br>
                            Original Credit Amount: ${window.currentSaleCreditInfo.original_credit_amount.toFixed(2)}<br>
                            Paid Amount: ${window.currentSaleCreditInfo.paid_amount.toFixed(2)}<br>
                            <strong>Available Credit Balance: ${window.currentSaleCreditInfo.credit_balance.toFixed(2)}</strong><br>
                            <small class="text-muted">Good condition items can be returned regardless of credit balance (including 0.00). Damaged/opened items require sufficient credit balance.</small>
                        `;
                        
                        // Insert credit info before the table
                        const tableContainer = document.querySelector('#refundItemsTable').parentElement;
                        tableContainer.insertBefore(creditInfoDiv, document.querySelector('#refundItemsTable'));
                    } else {
                        refundMaxAmount = parseFloat(data.sale_total || data.total_amount || 0);
                    }
                    
                    // Show payment breakdown if multiple payment methods
                    if (data.has_multiple_payments) {
                        const paymentInfoDiv = document.createElement('div');
                        paymentInfoDiv.className = 'alert alert-warning mb-3';
                        
                        let paymentHtml = '<strong>Payment Breakdown:</strong><br>';
                        data.payment_methods.forEach(p => {
                            const icon = getPaymentMethodIcon(p.method);
                            if (p.is_debt) {
                                paymentHtml += `${icon} ${p.method} (Debt): ${p.amount.toFixed(2)}<br>`;
                            } else {
                                paymentHtml += `${icon} ${p.method}: ${p.amount.toFixed(2)}<br>`;
                            }
                        });
                        
                        paymentHtml += `<strong>Total Paid: ${data.total_paid.toFixed(2)}</strong>`;
                        if (data.credit_amount > 0) {
                            paymentHtml += `<br><strong>Credit Debt: ${data.credit_amount.toFixed(2)}</strong>`;
                        }
                        paymentHtml += '<br><small class="text-muted">Refund will be processed according to the original payment methods.</small>';
                        
                        paymentInfoDiv.innerHTML = paymentHtml;
                        
                        // Insert payment info before the table
                        const tableContainer = document.querySelector('#refundItemsTable').parentElement;
                        tableContainer.insertBefore(paymentInfoDiv, document.querySelector('#refundItemsTable'));
                    }
                    
                    tbody.innerHTML = '';
                    if (saleItems.length === 0) {
                        tbody.innerHTML = '<tr><td colspan="10" class="text-center">No items found for this sale</td></tr>';
                        return;
                    }
                    
                    saleItems.forEach(item => {
                        const tr = document.createElement('tr');
                        tr.dataset.saleItemId = item.id;
                        tr.dataset.productId = item.product_id;
                        
                        // Add styling for returned items and grey out items that can't be returned
                        if (item.is_returned) {
                            tr.classList.add('item-returned');
                            tr.classList.add('item-disabled');
                            tr.style.opacity = '0.6';
                        }
                        if (!item.can_return) {
                            tr.classList.add('item-disabled');
                            tr.style.opacity = '0.6';
                        }
                        
                        const checkboxDisabled = (!item.can_return || item.is_returned);
                        
                        tr.innerHTML = `
                            <td><input type="checkbox" class="item-checkbox" ${checkboxDisabled ? 'disabled' : ''} 
                                       ${!item.can_return ? 'title="This item cannot be returned (older than 7 days)"' : ''}
                                       ${item.is_returned ? 'title="This item has already been returned"' : ''}></td>
                            <td>${item.name}</td>
                            <td class="serial-number-cell">
                                ${item.serial_numbers && item.serial_numbers.length > 0 ? 
                                    item.serial_numbers.map(sn => sn.serial_number).join(', ') : 
                                    '<span class="text-muted">No Serial</span>'
                                }
                            </td>
                            <td>
                                <input type="number" name="quantity" class="form-control form-control-sm" 
                                       min="1" max="${item.remaining_quantity || item.quantity}" value="${item.remaining_quantity || item.quantity}" 
                                       style="width: 80px;" ${checkboxDisabled ? 'disabled' : ''}>
                            </td>
                            <td>
                                <input type="number" name="unit_price" class="form-control form-control-sm" 
                                       value="${item.unit_price}" readonly style="width: 100px;">
                            </td>
                            <td>
                                <input type="number" name="restock_quantity" class="form-control form-control-sm" 
                                       min="0" max="${item.remaining_quantity || item.quantity}" value="${item.remaining_quantity || item.quantity}" 
                                       style="width: 80px;" ${checkboxDisabled ? 'disabled' : ''}>
                            </td>
                            <td>
                                <select name="condition" class="form-select form-select-sm" style="width: 120px;" ${checkboxDisabled ? 'disabled' : ''}>
                                    <option value="Good">Good</option>
                                    <option value="Damaged">Damaged</option>
                                    <option value="Opened">Opened</option>
                                    <option value="Other">Other</option>
                                </select>
                            </td>
                            <td class="item-total-cell">
                                <input type="number" name="manual_total" class="form-control form-control-sm d-none" min="0" max="${item.quantity * item.unit_price}" step="0.01" style="width: 100px;">
                                <span class="item-total">${(item.quantity * item.unit_price).toFixed(2)}</span>
                            </td>
                            <td class="discount-cell">
                                <span class="discount-amount">${(item.discount || 0).toFixed(2)}</span>
                            </td>
                            <td class="pos-price-cell">
                                <span class="pos-price">${((item.quantity * item.unit_price) - (item.discount || 0)).toFixed(2)}</span>
                                <br>                                <small class="text-muted payment-method">
                                    ${data.has_multiple_payments ? 
                                        data.payment_methods.map(p => 
                                            `${getPaymentMethodIcon(p.method)} ${p.method}${p.is_debt ? ' (Debt)' : ''}: ${p.amount.toFixed(2)}`
                                        ).join('<br>') :
                                        `${getPaymentMethodIcon(data.payment_method || 'Cash')} ${data.payment_method || 'Cash'}`
                                    }
                                </small>
                            </td>
                        `;
                        tbody.appendChild(tr);

                        // Attach event listeners for this row
                        const conditionSelect = tr.querySelector('select[name="condition"]');
                        const manualTotalInput = tr.querySelector('input[name="manual_total"]');
                        const itemTotalSpan = tr.querySelector('.item-total');
                        const quantityInput = tr.querySelector('input[name="quantity"]');
                        const unitPriceInput = tr.querySelector('input[name="unit_price"]');
                        const restockInput = tr.querySelector('input[name="restock_quantity"]');

                        function updateTotalField() {
                            const condition = conditionSelect.value;
                            const qty = parseFloat(quantityInput.value) || 0;
                            const price = parseFloat(unitPriceInput.value) || 0;
                            const maxTotal = qty * price;
                            manualTotalInput.max = maxTotal;
                            if (["Damaged", "Opened", "Other"].includes(condition)) {
                                manualTotalInput.classList.remove('d-none');
                                manualTotalInput.value = itemTotalSpan.textContent;
                                itemTotalSpan.classList.add('d-none');
                                // Set restock qty to 0 and disable
                                restockInput.value = 0;
                                restockInput.disabled = true;
                            } else {
                                manualTotalInput.classList.add('d-none');
                                itemTotalSpan.classList.remove('d-none');
                                itemTotalSpan.textContent = (qty * price).toFixed(2);
                                // Set restock qty to match refund qty and enable
                                restockInput.value = qty;
                                restockInput.disabled = false;
                            }
                            updateRefundTotals();
                        }
                        // Real-time validation for manual total
                        manualTotalInput.addEventListener('input', function() {
                            const qty = parseFloat(quantityInput.value) || 0;
                            const price = parseFloat(unitPriceInput.value) || 0;
                            const maxTotal = qty * price;
                            if (parseFloat(this.value) > maxTotal) {
                                this.value = maxTotal.toFixed(2);
                            }
                            updateRefundTotals();
                        });
                        // Initial state
                        updateTotalField();
                        // Event listeners for this row
                        conditionSelect.addEventListener('change', updateTotalField);
                        quantityInput.addEventListener('input', updateTotalField);
                        manualTotalInput.addEventListener('input', updateRefundTotals);
                        
                        // Add checkbox event listener
                        const checkbox = tr.querySelector('.item-checkbox');
                        checkbox.addEventListener('change', updateRefundTotals);
                    });
                    
                    // Select all checkbox - only select eligible items
                    const selectAllCheckbox = document.getElementById('selectAllItems');
                    selectAllCheckbox.addEventListener('change', function() {
                        document.querySelectorAll('.item-checkbox:not(:disabled)').forEach(checkbox => {
                            checkbox.checked = this.checked;
                        });
                        updateRefundTotals();
                    });
                    
                    // Reset select all checkbox state
                    selectAllCheckbox.checked = false;
                    selectAllCheckbox.indeterminate = false;

                    // Initialize refund total
                    updateRefundTotals();
                } else {
                    tbody.innerHTML = `<tr><td colspan="9" class="text-center text-danger">Error: ${data.message || 'Failed to load sale items'}</td></tr>`;
                }
            })
            .catch(error => {
                console.error('Error fetching sale items:', error);
                tbody.innerHTML = `<tr><td colspan="9" class="text-center text-danger">Error loading sale items: ${error.message}</td></tr>`;
            });
    }

    function viewRefundDetails(refundId) {
        // Fetch refund details
        fetch(`../api/get_refund_details.php?id=${refundId}`)
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    const refund = data.refund;
                    const items = data.items;
                    
                    // Populate modal
                    document.getElementById('refundDetailsModal').querySelector('.modal-title').textContent = 
                        `Refund Details - #${refund.refund_id}`;
                    
                    const detailsBody = document.getElementById('refundDetailsBody');
                    detailsBody.innerHTML = `
                        <div class="table-responsive">
                            <table class="table table-bordered">
                                <tr>
                                    <th>Refund ID</th>
                                    <td>${refund.refund_id}</td>
                                    <th>Date</th>
                                    <td>${refund.refund_date}</td>
                                </tr>
                                <tr>
                                    <th>Original Sale ID</th>
                                    <td>${refund.sale_id}</td>
                                    <th>Amount</th>
                                    <td>Ksh ${parseFloat(refund.refund_amount).toFixed(2)}</td>
                                </tr>
                                <tr>
                                    <th>Reason</th>
                                    <td colspan="3">${refund.reason}</td>
                                </tr>
                            </table>
                            
                            <h5 class="mt-4">Refunded Items</h5>
                            <table class="table table-bordered">
                                <thead>
                                    <tr>
                                        <th>Item</th>
                                        <th>Quantity</th>
                                        <th>Price</th>
                                        <th>Total</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    ${items.map(item => `
                                        <tr>
                                            <td>${item.item_name}</td>
                                            <td>${item.quantity}</td>
                                            <td>Ksh ${parseFloat(item.price).toFixed(2)}</td>
                                            <td>Ksh ${parseFloat(item.total).toFixed(2)}</td>
                                        </tr>
                                    `).join('')}
                                </tbody>
                            </table>
                        </div>
                    `;
                    
                    // Show modal
                    const modal = new bootstrap.Modal(document.getElementById('refundDetailsModal'));
                    modal.show();
                } else {
                    alert('Error loading refund details: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error loading refund details');
            });
    }

    function showCreditDetails(saleId) {
        const modalBody = document.querySelector('#creditDetailsModal .modal-body #creditDetailsContent');
        modalBody.innerHTML = '<div class="text-center"><div class="spinner-border text-primary" role="status"></div><p class="mt-2">Loading credit details...</p></div>';

        const modal = new bootstrap.Modal(document.getElementById('creditDetailsModal'));
        modal.show();

        fetch(`../api/get_credit_details.php?sale_id=${saleId}`)
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    const credit = data.credit;
                    const payments = data.payments;
                    const items = data.items || [];
                    let html = `
                        <div>
                            <h5>Credit Sale Information</h5>
                            <table class="table table-sm">
                                <tr><th>Sale ID:</th><td>${credit.sale_id}</td></tr>
                                <tr><th>Invoice #:</th><td>${credit.invoice_number}</td></tr>
                                <tr><th>Date:</th><td>${credit.created_at}</td></tr>
                                <tr><th>Customer:</th><td>${credit.customer_name || 'Walk-in'}</td></tr>
                                <tr><th>Amount:</th><td>${parseFloat(credit.amount).toLocaleString(undefined, {minimumFractionDigits:2})}</td></tr>
                                <tr><th>Status:</th><td>${credit.credit_status || credit.status}</td></tr>
                            </table>
                            <h6>Credit Items</h6>
                            <div class="table-responsive">
                                <table class="table table-bordered table-sm">
                                    <thead>
                                        <tr>
                                            <th>Item</th>
                                            <th>Serial No</th>
                                            <th>Qty</th>
                                            <th>Unit Price</th>
                                            <th>Total</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        ${items.map(item => `
                                            <tr>
                                                <td>${item.product_name}</td>
                                                <td>${item.serial_no || ''}</td>
                                                <td>${item.quantity}</td>
                                                <td>${parseFloat(item.unit_price).toLocaleString(undefined, {minimumFractionDigits:2})}</td>
                                                <td>${(item.quantity * item.unit_price).toLocaleString(undefined, {minimumFractionDigits:2})}</td>
                                            </tr>
                                        `).join('')}
                                    </tbody>
                                </table>
                            </div>
                            <h6>Payment History</h6>
                            <ul style="padding-left:18px;">
                                ${payments.length > 0 ? payments.map(p => `<li>${p.created_at}: <strong>${parseFloat(p.amount).toLocaleString(undefined, {minimumFractionDigits:2})}</strong> (${p.payment_method}) by ${p.received_by}</li>`).join('') : '<li>No payments yet.</li>'}
                            </ul>
                        </div>
                    `;
                    modalBody.innerHTML = html;
                } else {
                    modalBody.innerHTML = `<div class="alert alert-danger">${data.message}</div>`;
                }
            })
            .catch(error => {
                console.error('Error:', error);
                modalBody.innerHTML = '<div class="alert alert-danger">Error loading credit details</div>';
            });
    }
    
    // Add event listener for preview button
    document.addEventListener('DOMContentLoaded', function() {
        const previewBtn = document.getElementById('previewNotificationBtn');
        if (previewBtn) {
            previewBtn.addEventListener('click', previewNotificationMessage);
        }
    });
</script>

<?php 
// Add permission modal components
echo getPermissionModalCSS();
echo getPermissionModalHTML();
echo getPermissionModalJS();
?>

<?php echo getUniversalPermissionModalHTML(); ?>
<?php echo getUniversalPermissionModalJS(); ?>
</body>
</html>